;;
;; Global variables.
;;
globals [count_MS0 count_MS1 ticks_MS01]

;;
;; turtle/breed.
;;

;;
;; turtle/breed variables.
;;
turtles-own [
  MS  ;; mental state
  OB  ;; observed behavior
  OBD ;; observed behavior discrepancy
]

;;
;; setup.
;;
to setup
  clear-all
  reset-ticks
  setup-world
  setup-turtles
  ;; etc
end

;;
;; setup-world.
;;
to setup-world
  ask patches [ set pcolor green + 2]
end

;;
;; setup-turtles.
;;
to setup-turtles
  ;; Initialize global variables.

  ;; Create the turtles and set their initial states.
  create-turtles number_agents
  [ set shape "default"
    setxy random-xcor random-ycor]

  ;; Initial MS.
  ask turtles
  [
    ifelse random 100 < MS1
    [ set MS 1
      set color 9.9]
    [ set MS 0
      set color 0  ]
  ]

  ;; Update counts.
  update-counts

end

;;
;; go.
;;
to go
  ;; Criteria to stop.
  if (ticks > 5000) [stop]
  if (count_MS0 = count_MS1) [stop]

  ;; Otherwise do-things.

  ;; Update status.
  update-turtles

  ;; tick.
  tick
end

;;
;; update-turtles.
;;
to update-turtles
  ask turtles
  [ ;; update MS after wear
    let WR (WR0 + (WR1 - WR0) * MS)
    if random-float 100 < WR
    [ ifelse MS = 0 [set MS 1] [set MS 0]]

    ;; update OB
    ifelse (MS = 1)
    [ set OB sqrt(random-float 1.)]
    [ set OB (1 - sqrt(1 - (random-float 1.)))]

    ;; update OBD
    set OBD (MS - mean [OB] of other n-of K turtles)

    ;; update MS after OBD
    ;; triangular distribution for given MS:
    let x0 MS - 1
    let x1 MS
    let xm 2 * MS - 1 ;; position of the mode
    let ym (PC0 + (PC1 - PC0) * MS) * 2
    ;; compute y (given MS and OBD):
    let y 0
    ifelse (OBD < xm)
    [
      ifelse ( xm = x0)
      [ set y ym]
      [ set y 0 + (OBD - x0) * ym / (xm - x0)]
    ]
    [
      ifelse ( xm = x1)
      [ set y ym]
      [ set y 0 + (x1 - OBD) * ym / (x1 - xm)]
    ]
    ;; Change MS according to y
    if random-float 100 < y * 0.1 ;; 0.1 is the interval precision width for OBD.
    [ ifelse MS = 0 [set MS 1] [set MS 0]]

    ;; Update counts.
    update-counts

    ;; update color
    ifelse MS = 0
    [set color 0]
    [set color 9.9]
  ]
end

to update-counts
  ;; Update counts.
  set count_MS0 count turtles with [MS = 0]
  set count_MS1 count turtles with [MS = 1]
end
@#$#@#$#@
GRAPHICS-WINDOW
340
10
777
448
-1
-1
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
270
10
333
43
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
270
50
333
83
NIL
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

BUTTON
270
90
333
123
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

SLIDER
15
10
135
43
number_agents
number_agents
0
1000
100.0
1
1
NIL
HORIZONTAL

SLIDER
140
10
260
43
MS1
MS1
0
100
80.0
5
1
%
HORIZONTAL

SLIDER
15
50
135
83
WR0
WR0
0
1
0.05
0.05
1
%
HORIZONTAL

SLIDER
140
50
260
83
WR1
WR1
0
1
0.2
.05
1
%
HORIZONTAL

PLOT
785
10
985
160
MS_count %
ticks
%
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"MS0" 1.0 0 -16777216 true "" "plot ifelse-value (count turtles = 0) \n[ 0 ]\n[count_MS0 / number_agents * 100.]"
"MS1" 1.0 0 -7500403 true "" "plot ifelse-value (count turtles = 0) \n[ 0 ]\n[count_MS1 / number_agents * 100.]"

PLOT
785
170
985
320
Mean
ticks
NIL
0.0
10.0
0.0
1.0
true
true
"" ""
PENS
"OB" 1.0 0 -16777216 true "" "plot ifelse-value (count turtles = 0) \n[ 0 ]\n[ mean [OB] of turtles ]"
"MS" 1.0 0 -2674135 true "" "plot ifelse-value (count turtles = 0) \n[ 0 ]\n[ mean [MS] of turtles ]"
"OBD" 1.0 0 -7500403 true "" "plot ifelse-value (count turtles = 0) \n[ 0 ]\n[ mean [OBD] of turtles ]"

SLIDER
15
130
135
163
K
K
0
100
30.0
5
1
NIL
HORIZONTAL

SLIDER
15
90
135
123
PC0
PC0
0
10
0.5
.1
1
%
HORIZONTAL

SLIDER
140
90
260
123
PC1
PC1
0
10
2.5
.1
1
%
HORIZONTAL

MONITOR
785
330
845
375
MS0 %
(count_MS0 / number_agents) * 100.
8
1
11

MONITOR
850
330
907
375
ticks
ticks
17
1
11

@#$#@#$#@
## WHAT IS IT?

This model simulates how a feature (a behavior) distributes (spreads) within a population.

Two aspects are differenciated:

  1. A mental state (MS), which is what the agent (turtle) thinks about the behavior. For each agent the MS is represented as a discrete (0/1) variable.
  2. An observed behavior (OB), which is what the agent performs in the community. For each agent the OB is represented as a continuous variable in the range [0, 1].

There is a relationship between MS and OB, of probabilistic nature. The notion that MS = 0 (1) should tend to yield OB = 0 (1) is respected.

MS spreads by two simultaneous mechanisms:

  1. By wear: the agents somehow "get tired" of their MS and shift to the other.
  2. By socialization: the agents observe the behavior (OB) of other agents, notice the discrepancy between their own MS and the average OB of their peers, and then adapt their MS in order to reduce the discrepancy. The number of peers to be observed by the agent is controlled from a slider (K).

From an initial distribution of MS among the agents, the model shows the changes in the distribution (spread) of MS in time.


## HOW IT WORKS

The model differentiates the agents by their MS: all the parameters that control their actions are defined for the extreme cases MS = 0 and MS = 1. The values of these parameters are defined by sliders.

After initialization, the core steps followed by the model are as follows:

  1. **Update MS after wear.** An average "wear rate" is calculated for the agent based on his MS, and then a calculation determines whether he will change or maintain his MS. WR0 (1) is the probability that if MS = 0 (1) the agent will change his MS because of "wear" in one tick-interval.

  2. **Update OB.** The agent's OB depends on MS, but has "some noise" on it. A triangular relationship between MS and OB is assumed: if MS = 0, then the most probable outcome is OB = 0, and the less probable is OB = 1, with a linear trend in between. For MS = 1 the most probable outcome is OB = 1, etc. 

  3. **Update OBD.** K agents are selected randomly and its average OB is deducted from the agent's MS.

  4. **Update MS based on OBD.** The larger the OBD (in absolute value), the larger is the probability that MS will change. For a given MS, this probability of change follows a triangular distribution. PC0 (1) is the total probability that if MS = 0 (1) the agent will change its MS due to OBD. 


Additionally on each tick:

  1. The counters for each type of MS are updated.
  2. The colors of teh agents are also updated. (MS = 0 is black, MS = 1 is white).


## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)
Items in the Interface:

  1. number_agents, slider: number of agents (turtles).
  2. MS1, slider: percentage of agents with MS = 1 at start.
  3. WR0 (1), slider: the probability that if MS = 0 (1) the agent will change his MS because of "wear" in one tick-interval.
  4. PC0 (1), slider: the total probability that if MS = 0 (1) the agent will change its MS due to OBD.
  5. K, slider: the number of agents randomly selected to asses their OB and compare it with the agent's MS.


## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

  1. What if the agent observes the K-peers with bias? The extracted parameter could be the average (or a quantile) after removing from the sample the agents whose OB is more distant from the agent's.

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.1.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
1
@#$#@#$#@
