extensions [rngs]
breed [socials social]                 ;all of these fish will want high conspecific densities
breed [heterogeneouss heterogeneous]   ;this population will want a wide range of conspecific densities
breed [asocials asocial]               ;all of these fish will want low conspecific densities
breed [e_asocials e_asocial]           ;these are extremely asocial fish
breed [e_socials e_social]             ;these are extremely social fish
breed [Controls Control]               ;these fish lack any behavioral information
breed [Nulls null]                     ;these fish have total random personalities with equal probability of every value between 0 and 1
breed [Bimodals bimodal]               ;creates a population where half are social and half are asocial
turtles-own
[
  SOCIABILITY                          ;state variable bound between 0 and 1
  TARGET_DENSITY                       ;dependent on sociability
  MY_DENSITY                           ;full knowledge of local conspecific density
  SOCIAL_RANK                          ;dependent on sociability: bottom up ranked sociability
  MAX_SEARCH_RADIUS                    ;dependent on Max_moves_per_tick
  PANIC_TIME                           ;dependent on dispersal window (10%)
  TIME_SINCE_LAST_POOL_FOUND           ;counter which tracks how many days have elapsed since the fish last found a pool
  DAYS_UNHAPPY                         ;counter which tracks how many days have elapsed since the fish has encountered a suitable density
  LAST_POOL_FOUND                      ;patch identifier: coordinates of the last pool the fish evaluated
  BEEN_HERE_BEFORE?                    ; boolean identifier set to true if the fish has been in that pool before and is evaluating it's density
  FIRST_PATCH                          ;patch identifier: coordinates the fish started at
  LAST_PATCH                           ;patch identifier: coordinates the fish was last at
  TOTAL_DISTANCE_TRAVELED              ;cummulative distance traveled from last-patch to current coordinates, added on every tick
  LINEAR_DISTANCE_TRAVELED             ;distance from current patch to first patch on the final tick
  TIME_SPENT_HERE                      ;counter which tracks how many days a turtle has been in a pool
  STAY?                                ;boolean identifier: set to true if the fish is either happy or waiting for conditions to improve in the pool
  HANG_OUT_COUNTER                     ;dependent on sociability: counts down by 10% every day the fish is unhappy
]

patches-own
[
  DEPTH                                ;state variable
  DENSITY                              ;density of fish at the start of the day
  MEAN_SOCIABILITY                     ;mean sociability of turtles on the patch, called by sociability list
  VARIANCE_LIST                        ;variance of sociability turtles in the hole
  STDEV_LIST                           ;standard deviation of sociability turtles in the hole
]

globals
[
  ALPHA                                ;beta distribution parameter
  BETA                                 ;beta distribution parameter
  DIST_COUNT                           ;count of sociability distributions activated for error checking purposes
  RANDOM_FISH                          ;a random currently living fish
  START#                               ;the number of fish at day 1
  CURRENT_COUNT                        ; the current number of living fish
  SOCIAL_RANK_LIST                     ;social ranking (lowest to highest sociability scores)
  DISTANCE_LIST                        ;list of sociabilities and associated total distances traveled
                                       ;DISPERSAL_WINDOW                    ;;DEFINED ON INTERFACE= total run time
                                       ;MAX_TARGET                          ;;DEFINED ON INTERFACE= maximum preferred density if sociability = 1
                                       ;MAX_MOVES_PER_TICK                  ;;DEFINED ON INTERFACE= maximum linear movement distance of a single fish in a single da+y
                                       ;START_FISH_COUNT                    ;;DEFINED ON INTERFACE= initial population size to be generated
  CUT_RUN_SHORT?
  DENSITY_LIST                         ;LIST OF X COORDINATE AND ASSOCIATED DENSITIES and densities
  SOCIABILITY_LIST                     ;x coordinate, y coordinate and mean sociability of turtles in the patch
]


to setup
  clear-all
  reset-ticks
  stop-inspecting-dead-agents
  set_globals                                                               ;if using optimal settings this will set all global factors to those used in experiments
  make_pools
  make_fish
  if Show_social_v_asocial_path? = TRUE
  [ if user-yes-or-no? ["To use this option you must select only the Control or Hetero distributions in the panel on the left, please ensure you have done so before proceeding"]
    [
      ask one-of turtles with [sociability > 0.9] [set color blue set size 4 pen-down ask other turtles with [sociability > 0.5] [set size 0]]
      ask one-of turtles with [sociability < 0.1] [set color red set size 4 pen-down ask other turtles with [sociability < 0.6] [set size 0]]
    ]
  ]

 ; ifelse EXPERIMENT_SETTINGS? = FALSE
 ; [update_plots]                                                            ;if not running an experiment then update the plots in the window, but don't if experimenting (speed optimization)
 ; [ask turtles [set size 0]]                                                ;if experimenting, make all turtles invisible (speed optimization)

  ask patches [set DENSITY count turtles-here]
  ; ask one-of patches [set DENSITY_LIST list "," DENSITY_LIST]

  set social_rank_list sort-on [sociability] turtles                        ;sort the social rank by sociability low to high
  Look_at_one_fish                                                          ;if watch_a_random_fish? is activated this makes all other fish invisible and tracks one random fish
  set START# count turtles
  fish_initial_state                                                        ;sets all turtle owned parameters to their initial state for all fish

end

to go
  tick
  if count turtles <= 10 [stop]                                             ;lines 2 and 3 of Go are error checking protocols to stop any runs that are using undesired scenarios (experiment speed optimization)
  if DIST_COUNT > 1 [ask n-of (count turtles - 1) turtles [die]]

  ask patches [set DENSITY count turtles-here]
  ask turtles [set MY_DENSITY count turtles-here]

  if Show_social_v_asocial_path? = TRUE
  [
    ask turtles with [size > 1] [set label precision TOTAL_DISTANCE_TRAVELED 2]
  ]


  ifelse Control? = TRUE
    [ask Controls [Control_pool_search]]                                    ;if running a Control scenario, then tell Controls to search for pools without using social information
    [  ifelse SOCIAL_RANK_ordered_movement? = TRUE                          ;if you want to use social rank ordered movement, then make the list and tell the fish to search for a pool in order of that list
      [let rank-list sort-on [SOCIAL_RANK] turtles
        foreach rank-list
        [ask ? [pool_search]]]
      [ask turtles [pool_search]]                                           ;if not, then just tell the fish to search for a pool in a random order
    ]

  ask turtles [track_travel]                                                ;this keeps track of how far each fish has moved

  if ticks >= DISPERSAL_WINDOW [  Final_tick_procedures
    stop]                                    ;stops the run and updates the plots/outputs when the dispersal window closes


  if watch_a_RANDOM_FISH? = TRUE                                            ;if you're watching one fish, this block tells it to highlight it's search radius each day with a white splotch
    [ask turtle RANDOM_FISH [show MY_DENSITY ask patches in-radius MAX_SEARCH_RADIUS [set pcolor white ]]]

  if CUT_RUN_SHORT? = TRUE [Final_tick_procedures
    stop]
  update_plots
end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;movement protocols;;;;;;;;;;;movement protocols;;;;;;;;;;;;;movement protocols;;;;;;;movement protocols;;;;;;;;;;;;;;;;;movement protocols;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to Control_pool_search

  let D min [DEPTH] of patches in-radius MAX_SEARCH_RADIUS                                                     ;minimum DEPTH of all patches reachable in one day
  let Dn min-one-of patches in-radius MAX_SEARCH_RADIUS [DEPTH]                                                ;patch with the lowest DEPTH which can be reached in one day
  let LDe [DEPTH] of patch-here                                                                                ;current patch DEPTH

  ifelse patch-here = LAST_POOL_FOUND                                                                          ;note if you've found the same pool that you most recently evaluated
    [set BEEN_HERE_BEFORE? TRUE ]
    [set BEEN_HERE_BEFORE? FALSE ]


  ifelse DEPTH < 0                                                                                             ;if you find a pool then reset your counter, if not then add to it
    [set TIME_SINCE_LAST_POOL_FOUND 0]
    [set TIME_SINCE_LAST_POOL_FOUND TIME_SINCE_LAST_POOL_FOUND + 1]

  ifelse DEPTH < 0                                                                                             ;if you're in a POOL
    [move-to patch-here                                                                                        ;center yourself in it
      ifelse ((random-float 101) > 50)
      [set color yellow stop]
      [set color pink new_pool_search]                                                                                                     ;stop moving
    ]

    [ifelse LDe > D AND D < 0                                                                                  ;if your current patch DEPTH is greater than the minimum DEPTH you could achieve in one day
      [move-to one-of patches in-radius MAX_SEARCH_RADIUS with-min [DEPTH]
        stop
      ]                                                                                                        ;go the the deeper patch and stop moving
      [new_pool_search]                                                                                        ;if not, then go find a pool
    ]
end

to new_pool_search
  ifelse TIME_SINCE_LAST_POOL_FOUND < PANIC_TIME                                                               ;is it time to panic about how long it has been since you found a pool?
    [right random-float SMALL_SEARCH_RADIUS                                                                    ;if not, then keep searching your local area intensively (you found a pool here before, there's probably another one nearby right?)
      forward random-float MAX_MOVES_PER_TICK
    ]
    [right random-float BIG_SEARCH_RADIUS                                                                      ;if you are panicking then start a broader search pattern (you're not having any luck here so lets expand the search area)
      forward random-float MAX_MOVES_PER_TICK
    ]
end

to pool_search

  let D min [DEPTH] of patches in-radius MAX_SEARCH_RADIUS                                                     ;minimum DEPTH of all patches reachable in one day
  let Dn min-one-of patches in-radius MAX_SEARCH_RADIUS [DEPTH]                                                ;patch with the lowest DEPTH which can be reached in one day
  let LDe [DEPTH] of patch-here                                                                                ;current patch DEPTH



  ifelse patch-here = LAST_POOL_FOUND                                                                          ;note if you're in the last pool you abandoned or not
  [set BEEN_HERE_BEFORE? TRUE ]
  [set BEEN_HERE_BEFORE? FALSE ]

  ifelse patch-here = LAST_PATCH                                                                               ;if you haven't moved today, add one to your counter. if you have, then set that counter to 0
  [set TIME_SPENT_HERE TIME_SPENT_HERE + 1]                                                                    ;this counter is used in the hang out probability calculation
  [set TIME_SPENT_HERE 0]

  ifelse DEPTH < 0                                                                                             ;if you're in a pool then set that counter to 0, if not then add a day to it
    [set TIME_SINCE_LAST_POOL_FOUND 0]
    [set TIME_SINCE_LAST_POOL_FOUND TIME_SINCE_LAST_POOL_FOUND + 1]



  ifelse DEPTH < 0                                                                                             ;if you're in a POOL
    [move-to patch-here                                                                                        ;center yourself in it
      should_I_hang_out                                                                                        ;decide if you should stay or not
      ifelse stay? = TRUE                                                                                      ;if you decide to stay then center yourself and stop for the day
      [move-to patch-here
        stop
      ]

      [set LAST_POOL_FOUND patch-here                                                                          ;if you decided to move on then note the location of the pool and search for a new one
        new_pool_search
      ]
    ]

    [ifelse LDe > D AND D < 0                                                                                  ;if your current patch DEPTH is greater than the minimum DEPTH you could achieve in one day
      [move-to Dn
        stop
      ]                                                                                                        ;go the the deeper patch and stay there for the day

      [new_pool_search]
    ]

end


to should_I_hang_out                                                                                          ;this calculates the STAY? factor as a function of time, sociability and patch suitability

  let acceptable ((DENSITY >= (TARGET_DENSITY * (1 + DENSITY_WIGGLE_ROOM))) AND DENSITY < (TARGET_DENSITY * (1 - DENSITY_WIGGLE_ROOM))) ;be happy if you're in a patch that's within your the wiggle room range of density


  if TIME_SPENT_HERE < 2[  set HANG_OUT_COUNTER (MAX_HANG_OUT_TIME * SOCIABILITY)]                            ;if you just got there, set your hang out counter to the individual's maximum possible value

  move-to patch-here                                                                                          ;center yourself in the patch

  ifelse Acceptable
  [set stay? TRUE                                                                                             ;if you're happy then stick around and reset your hang out counter
    set HANG_OUT_COUNTER (MAX_HANG_OUT_TIME * SOCIABILITY)
    stop
  ]

  [ifelse (random-float MAX_HANG_OUT_TIME) > HANG_OUT_COUNTER                                                 ;if you're not happy then draw a random number.
                                                                                                              ;if it's greater than your hang out counter
    [set stay? FALSE                                                                                          ;note the location of your pool, leave, and max out your counter again to prepare for the next time
      set LAST_POOL_FOUND patch-here

      set HANG_OUT_COUNTER (MAX_HANG_OUT_TIME * SOCIABILITY)
    ]

    [set HANG_OUT_COUNTER (HANG_OUT_COUNTER * DECAY_COUNTER)                                                  ;if not then deduct 1- the DECAY_COUNTER from your HANG_OUT_COUNTER and stay for the rest of the day
      set stay? TRUE
    ]
  ]

end





to wall_jump                                                                                                 ;this procedure makes fish turn away from world boundaries by having them draw an opposing random heading when they're within 5 patches of an edge
  if abs pxcor >= (max-pxcor - 5)
  [set heading (360 - (heading + random-float 90))]

  if abs pycor >= (max-pycor - 5)
  [set heading (270 - (heading + random-float 90))]

  if abs pxcor <= (min-pxcor + 5)
  [set heading (180 - (heading + random-float 90))]

  if abs pycor >= (min-pxcor + 5)
  [set heading (360 - (heading + random-float 360))]

end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;world creation;;;;;;;;;;;;;world creation;;;;;;;;;;;;;world creation;;;;;;;;;;world creation;;;;;;;;;;;;;;;;;world creation;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


to make_pools
  ifelse Grid_or_Rand_landscape? = TRUE
  [
    ask patches with [pxcor mod 20 = 0 and pycor mod 25 = 0  ]                                                               ;ask patches at every xy multiple of 25 to set their depth to -2
    [set DEPTH -2

    ]
    ask patches with [pxcor < 5] [set DEPTH 0]                                                                               ;set the source depth to 0

  ]


  [
    ;this block was used to generate a random landscape which was converted into a set csv file

    file-open "depth_for_rand_landscape.txt"
    while [not file-at-end?]
    [
      let next-X file-read
      let next-Y file-read
      let next-depth file-read
      ask patch next-x next-y [set depth next-depth]
    ]
  ]
    ask patches with [depth < 0] [set pcolor black]
    ask patches with [depth >= 0] [set pcolor blue + 2]
    ask patches with [pxcor < 5] [set pcolor blue - 2]


  file-close-all


end


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;world ends;;;;;;;;;;;;;;world ends;;;;;;;;;;;;;;world ends;;;;;;;;;;;world ends;;;;;;;;;;;;;;;;;;world ends;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to apocalypse_check                                                                                                          ;kill every fish not in a pool at the end of the dispersal window
  if ticks >= (DISPERSAL_WINDOW)
  [ ask turtles with [DEPTH >= 0]
    [die]
  ]

end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;turtle creation;;;;;;;;;;;;;turtle creation;;;;;;;;;;;;;turtle creation;;;;;;;;;;turtle creation;;;;;;;;;;;;;;;;;turtle creation;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



to source
  setxy (random-float 5) (random max-pycor)
end


to make_fish

;  if EXPERIMENT_SETTINGS? = TRUE ;THIS BLOCK OF CODE IS CALLED DURING EXPERIMENTS. IT REMOVES ALL AESTHETIC PROPERTIES OF THE FISH TO OPTIMIZE MODEL RUN SPEED
  ;[
    set DIST_COUNT 0                                                             ;this code block is used to count the number of SOCIABILITY distributions selected.
    if social? = true [set DIST_COUNT DIST_COUNT + 1]
    if asocial? = true [set DIST_COUNT DIST_COUNT + 1]
    if hetero? = true [set DIST_COUNT DIST_COUNT + 1]


    if Control? = true [set DIST_COUNT DIST_COUNT + 1]







    if Social? = TRUE
      [create-socials (START_FISH_COUNT / DIST_COUNT)
        ask socials
        [    set shape "fish"
           set color blue
          set size 3
          set ALPHA 10
          set BETA 2
          source
          rngs:init
          let stream_id random-float 9999
          let seed random-float 9999
          rngs:set-seed stream_id seed
          let dist rngs:rnd-beta stream_id ALPHA BETA
          set SOCIABILITY dist

        ]
      ]

    if hetero? = TRUE
    [create-heterogeneouss (START_FISH_COUNT / DIST_COUNT)
      ask heterogeneouss
        [
           set shape "fish"
           set color scale-color green sociability 0 1
          set size 3
          set ALPHA 2
          set BETA 2
          source
          rngs:init
          let stream_id random-float 9999
          let seed random-float 9999
          rngs:set-seed stream_id seed
          let dist rngs:rnd-beta stream_id ALPHA BETA
          set SOCIABILITY dist

        ]
    ]

    if asocial? = TRUE
    [create-asocials (START_FISH_COUNT / DIST_COUNT)
      ask asocials
        [    set shape "fish"
           set color red
          set size 3
          set ALPHA 2
          set BETA 10
          source
          rngs:init
          let stream_id random-float 9999
          let seed random-float 9999
          rngs:set-seed stream_id seed
          let dist rngs:rnd-beta stream_id ALPHA BETA
          set SOCIABILITY dist

        ]
    ]


    if Control? = TRUE
    [create-Controls (START_FISH_COUNT / DIST_COUNT)
      ask Controls
        [    set shape "fish"
           set color black
          set size 3
          set SOCIABILITY 1.1
          source
        ]
    ]
 ; ]





  ask turtles [
    set TARGET_DENSITY (1 + round (SOCIABILITY * MAX_TARGET))                         ;creates the individual target DENSITY value.
   ; set MAX_SEARCH_RADIUS (MAX_MOVES_PER_TICK / 4)
   set MAX_SEARCH_RADIUS (MAX_MOVES_PER_TICK * MAX_SEARCH_RADIUS2) ;used for sensitivity analysis, annotate out and set value in above line to optimal level
    set heading 90]
  ranking
  if DIST_COUNT = 0
    [ crt 1]



end


to ranking
  let rank-list sort-on [SOCIABILITY] turtles                                                                 ;create a low to high sorted list of ranked SOCIABILITY scores
  let ranks n-values length rank-list [ ? ]                                                                   ;rename each turtle according to it's ranked location in that list
    (foreach rank-list ranks [ask ?1 [set SOCIAL_RANK ?2] ] )
end

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;plots & counters;;;;;;;;;;;;;plots & counters;;;;;;;;;;;;;plots & counters;;;;;;;;;;plots & counters;;;;;;;;;;;;;;plots & counters;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

to update_plots

  set-current-plot "SOCIABILITY distribution"                                              ;draw a histogram of the population sociabilities
  histogram [SOCIABILITY] of turtles


  set-current-plot "Individual target DENSITY"                                             ;draw a histogram of the target densities of the population, if any fish are alive
  if count turtles > 0[
    set-plot-x-range 0 (round max [TARGET_DENSITY] of turtles + 1)
    histogram [TARGET_DENSITY] of turtles
  ]

  if ticks > 0                                                                             ;after tick 0
  [
    set-current-plot "Pool DENSITY"                                                        ;draw a histogram of the pool densities
    if (count (patches with [DEPTH < 0 AND DENSITY > 0])) > 0
      [
        set-plot-x-range 0 (max ([DENSITY] of patches with [DEPTH < 0 AND DENSITY > 0] ) )
        if(count (patches with [DEPTH < 0 AND DENSITY > 0 ]) > 0)
        [histogram [DENSITY] of patches with [DEPTH < 0 AND DENSITY > 0]]

      ]

  ]

  if ticks >= DISPERSAL_WINDOW
  [let happy (count turtles with [MY_DENSITY > TARGET_DENSITY * (1 - DENSITY_WIGGLE_ROOM) AND MY_DENSITY < TARGET_DENSITY * (1 + DENSITY_WIGGLE_ROOM)])

    set CURRENT_COUNT count turtles

    ;this block produces the output placed in the white output box on the interface

    output-print (word (( CURRENT_COUNT / START# ) * 100) "% of turtles survived")
    output-print (word " ")
    IFelse happy > 0
      [ output-print (word " ")
        output-print (word ((happy / CURRENT_COUNT ) * 100) ("% of survivors achieved desired DENSITY"))  ]
      [ output-print (word " ")
        output-print (word 0 ("% of survivors achieved desired DENSITY"))  ]

    if CURRENT_COUNT > 0
    [ output-print (word ((mean [LINEAR_DISTANCE_TRAVELED] of turtles) * 100) "(m) mean linear distance traveled " )
      output-print (word " ")
      output-print (word ((mean [TOTAL_DISTANCE_TRAVELED] of turtles) * 100) "(m) mean total distance traveled")
      output-print (word " ")
      output-print (word (mean ([DENSITY] of patches with [DENSITY > 0 AND DEPTH < 0])) " mean DENSITY of occupied patches")
      output-print (word " ")
      output-print (word (standard-deviation ([DENSITY] of patches with [DENSITY > 0 and DEPTH < 0])) "standard deviation of the DENSITY of occupied patches")
      output-print (word " ")
      output-print (word ((mean ([TARGET_DENSITY] of turtles)) ) "mean TARGET_DENSITY ")
      output-print (word " ")
      output-print (word (standard-deviation ([TARGET_DENSITY] of turtles)) "standard deviation of the target DENSITY of fish")
      output-print (word " ")
      output-print (word (median ([DENSITY] of patches with [DENSITY > 0 and DEPTH < 0])) " median DENSITY of occupied patches")
      output-print (word " ")
      output-print (word ((MEDIAN ([TARGET_DENSITY] of turtles)) ) "MEDIAN TARGET_DENSITY ")
      output-print (word " ")
      output-print (word (modes ([DENSITY] of patches with [DEPTH < 0 AND DENSITY > 0])) " DENSITY modes of occupied patches")
      output-print (word " ")
      output-print (word ((MODES [TARGET_DENSITY] of turtles) ) "MODAL TARGET_DENSITY ")
    ]
  ]

end



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;misc;;;;;;;;;;;;miscs;;;;;;;;;;;;;misc;;;;;;;;;;misc;;;;;;;;;misc;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
to set_globals
  if USE_OPTIMAL_SETTINGS? = TRUE                                                                     ;these values were selected using a sensativity analysis as they procude the desired patterns
  [set DENSITY_WIGGLE_ROOM 0.25
    set BIG_SEARCH_RADIUS 225
    set SMALL_SEARCH_RADIUS 133
    set MAX_HANG_OUT_TIME 5
    set SOCIAL_RANK_ORDERED_MOVEMENT? TRUE
    set WATCH_A_RANDOM_FISH? FALSE
    set START_FISH_COUNT 1000
    ; set DISPERSAL_WINDOW 200
    set MAX_TARGET 40
    set MAX_MOVES_PER_TICK 30
    set CUT_RUN_SHORT? FALSE
  ]

end

to Look_at_one_fish
  if watch_a_RANDOM_FISH? = TRUE
  [
    if watch_a_RANDOM_FISH?   = TRUE
    [
      ifelse user-yes-or-no? "Do you want to shrink all fish except the one of interest? They will continue to exist, but be invisible. This will allow the model to run faster, as well as make it easier to observe the fish of interest."
      [ask turtles [set size 0 ]

        set RANDOM_FISH random START_FISH_COUNT
        ask turtle RANDOM_FISH [set size 5 set color yellow pd]
        inspect turtle RANDOM_FISH
        output-print "Pull the inspection window down"
        output-print " "
        output-print "The white area around the fish you're watching is the"
        output-print " area searched in one day"
        output-print " "
        output-print "The command center shows this fish's local density"
        output-print "at the end of every day"
        output-print "............................................................... "
      ]
      [
        set RANDOM_FISH random START_FISH_COUNT
        ask turtle RANDOM_FISH [set size 5 set color yellow pd ]
        inspect turtle RANDOM_FISH
        output-print "Pull the inspection window down"
        output-print " "
        output-print "The white area around the fish you're watching is the"
        output-print " area searched in one day"
        output-print " "
        output-print "The command center shows this fish's local density"
        output-print "at the end of every day"
        output-print "................................................................. "

      ]
    ]
  ]

end

to fish_initial_state                                                                                                       ;this procedure is called during setup and sets all turtle parameters to the necessary initial state
  ask turtles
  [ set LAST_PATCH patch-here
    set FIRST_PATCH patch-here
    set TOTAL_DISTANCE_TRAVELED 0
    SET LINEAR_DISTANCE_TRAVELED 0
    Set TIME_SPENT_HERE 0
    Set STAY? FALSE
    set HANG_OUT_COUNTER 0
    set PANIC_TIME (DISPERSAL_WINDOW * PANIC_DISCOUNT)
  ]
end

to final_tick_procedures                                                                                                  ;this procedure is called at the end of the run to color fish based on their happieness and then start the apocalypse and update plots
  if Show_social_v_asocial_path? = TRUE
  [ask patches [set pcolor white]
    ask turtles with [size > 1]
      [
        ifelse sociability > 0.5
          [set label ""
            set plabel precision TOTAL_DISTANCE_TRAVELED 2
            ;ask one-of neighbors  [set plabel  "Social"]
            ask patch 60 60 [set plabel-color black set plabel "Social fish traveled an average (in m) of" ]
            ask patch 60 55 [set plabel-color black set plabel precision (100 * mean [TOTAL_DISTANCE_TRAVELED] of turtles with [sociability > 0.5]) 2]

          ]

          [set label ""
            set plabel precision TOTAL_DISTANCE_TRAVELED 2
            ;  ask one-of neighbors  [set plabel "Asocial"]
            ask patch 160 60 [set plabel-color black set plabel "Asocial fish traveled an average (in m) of" ]
            ask patch 160 55 [set plabel-color black set plabel precision (100 * mean [TOTAL_DISTANCE_TRAVELED] of turtles with [sociability < 0.5001]) 2]
          ]
      ]

  ]

  ask patches [set DENSITY count turtles-here]

  set DISTANCE_LIST (list "sociability distance")
  ask turtles [
    set DISTANCE_LIST lput "," DISTANCE_LIST
    set DISTANCE_LIST lput (precision SOCIABILITY 4) DISTANCE_LIST
    set DISTANCE_LIST lput " " DISTANCE_LIST
    set DISTANCE_LIST lput TOTAL_DISTANCE_TRAVELED DISTANCE_LIST
  ]

  ask one-of patches [set DENSITY_LIST (list "x density")]
  ask patches with [DEPTH < 0][
    set DENSITY_LIST lput " " DENSITY_LIST
    set DENSITY_LIST lput pxcor DENSITY_LIST
    set DENSITY_LIST lput " " DENSITY_LIST
    set DENSITY_LIST lput DENSITY DENSITY_LIST
  ]



  ask patches with [depth < 0 AND DENSITY > 0]  [set MEAN_SOCIABILITY (precision (mean [sociability] of turtles-here) 4)]
  ask patches with [DEPTH < 0 AND DENSITY = 0] [set MEAN_SOCIABILITY (0)]
  ask one-of patches [set SOCIABILITY_LIST (list "x y mean sociability")]
  ask patches with [DEPTH < 0][
    set SOCIABILITY_LIST lput "," SOCIABILITY_LIST
    set SOCIABILITY_LIST lput pxcor SOCIABILITY_LIST
    set SOCIABILITY_LIST lput " " SOCIABILITY_LIST
    set SOCIABILITY_LIST lput pYcor SOCIABILITY_LIST
    set SOCIABILITY_LIST lput " " SOCIABILITY_LIST
    set SOCIABILITY_LIST lput MEAN_SOCIABILITY SOCIABILITY_LIST
  ]



  ; ask patches with [DEPTH < 0] [set DENSITY_LIST (list (pxcor)(",") (DENSITY) (",") )]

;
;  ask patches with [DEPTH < 0 and DENSITY > 1] [set VARIANCE_LIST (list (pxcor)(",")(pycor) (",") (variance [SOCIABILITY] of turtles-here) (",") )]
;  ask patches with [DEPTH < 0 and DENSITY <= 1] [set VARIANCE_LIST (list (pxcor)(",")(pycor) (",") (0) (",") )]
;
;  ask patches with [DEPTH < 0 and DENSITY > 1] [set STDEV_LIST (list (pxcor)(",")(pycor) (",") (standard-deviation [SOCIABILITY] of turtles-here) (",") )]
;  ask patches with [DEPTH < 0 and DENSITY <= 1] [set STDEV_LIST (list (pxcor)(",")(pycor) (",") (0) (",") )]
;
  apocalypse_check
  set CURRENT_COUNT count turtles


end


to track_travel                                                                                                              ;this is invoked during go to tell fish to keep track of where they are with reference to where they just were as well as where they started
  if patch-here != LAST_PATCH
  [set TOTAL_DISTANCE_TRAVELED (TOTAL_DISTANCE_TRAVELED + (distance LAST_PATCH))
    set LAST_PATCH patch-here]
  set TOTAL_DISTANCE_TRAVELED precision TOTAL_DISTANCE_TRAVELED 4

  if ticks >= DISPERSAL_WINDOW [set LINEAR_DISTANCE_TRAVELED (distance FIRST_PATCH)]
  ; if Control? = TRUE AND ticks > 50 AND ((count turtles with [DEPTH < 0 ]) = (count turtles)) [set CUT_RUN_SHORT? TRUE]
end


to recolor_world
  ask patches with [depth >= 0] [set pcolor black]
  let MaxD (max [density] of patches with [depth < 0])
  ask patches with [depth < 0]
  [set plabel DENSITY
    ask other patches in-radius 5
    [set DENSITY (max [DENSITY] of patches in-radius 5 with [DEPTH < 0])
      set pcolor scale-color DENSITY red 0 MaxD
    ]
  ]
  ask patches with [depth < 0 AND density = 0] [ask other patches in-radius 5 [set pcolor white]]

  ask turtles [set size 0]

end

to recolor_world_by_sociability
  ask patches with [depth >= 0] [set pcolor black]
  ask patches with [DEPTH < 0 and DENSITY > 0] [set pcolor scale-color blue mean [SOCIABILITY] of turtles-here 0 1
    set plabel (precision (mean [sociability] of turtles-here) 2)
    let p pcolor
    ask patches in-radius 5 [set pcolor p] ]

  ask patches with [DEPTH < 0 AND DENSITY = 0]
  [
    set pcolor white
    ask other patches in-radius 5 [set pcolor white]
  ]

  ask turtles [set size 0]

end
@#$#@#$#@
GRAPHICS-WINDOW
207
11
875
372
-1
-1
3.274
1
10
1
1
1
0
0
0
1
0
200
0
100
0
0
1
day
30.0

BUTTON
5
10
69
43
NIL
Setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
75
10
138
43
NIL
Go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
146
10
209
43
Step
Go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
-3
45
95
78
Social?
Social?
1
1
-1000

SWITCH
-3
78
95
111
Hetero?
Hetero?
0
1
-1000

SWITCH
-1
113
95
146
Asocial?
Asocial?
1
1
-1000

PLOT
876
11
1089
131
Sociability distribution
NIL
NIL
0.0
1.0
0.0
1.0
true
false
"" ""
PENS
"" 0.01 1 -16777216 true "" ""

MONITOR
97
43
167
88
#_of_Fish
Count turtles
0
1
11

INPUTBOX
96
88
212
148
Dispersal_window
200
1
0
Number

INPUTBOX
96
150
211
210
Max_target
40
1
0
Number

PLOT
877
132
1088
252
Individual target density
NIL
NIL
-0.1
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" ""

INPUTBOX
96
211
211
271
Max_moves_per_tick
30
1
0
Number

SWITCH
406
513
620
546
Watch_a_random_fish?
Watch_a_random_fish?
1
1
-1000

INPUTBOX
96
272
210
332
Start_Fish_count
1000
1
0
Number

PLOT
877
250
1090
370
Pool density
Density
# of pools
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" ""

SLIDER
-2
406
174
439
Density_wiggle_room
Density_wiggle_room
0
1
0.25
0.01
1
NIL
HORIZONTAL

SWITCH
172
505
380
538
Social_rank_ordered_movement?
Social_rank_ordered_movement?
0
1
-1000

MONITOR
769
15
873
60
Number of pools
Count patches with [depth < 0]
0
1
11

MONITOR
770
64
872
109
Mean D of pools
mean [count turtles-here] of patches with [depth < 0]
0
1
11

SLIDER
171
405
378
438
Big_search_radius
Big_search_radius
0
360
225
1
1
NIL
HORIZONTAL

SLIDER
171
438
378
471
Small_search_radius
Small_search_radius
0
360
90
1
1
NIL
HORIZONTAL

SWITCH
0
146
95
179
Control?
Control?
1
1
-1000

SLIDER
171
471
378
504
MAX_HANG_OUT_TIME
MAX_HANG_OUT_TIME
0
Dispersal_window
3
1
1
NIL
HORIZONTAL

SWITCH
81
374
288
407
Use_optimal_settings?
Use_optimal_settings?
1
1
-1000

SWITCH
407
380
619
413
Grid_or_Rand_landscape?
Grid_or_Rand_landscape?
1
1
-1000

BUTTON
428
583
614
616
Color pool region by density
Recolor_world
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
427
639
613
672
Color pool region by sociability
recolor_world_by_sociability
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
408
444
622
477
Show_social_v_asocial_path?
Show_social_v_asocial_path?
1
1
-1000

SLIDER
-3
502
170
535
MAX_SEARCH_RADIUS2
MAX_SEARCH_RADIUS2
0
1
0.25
0.1
1
NIL
HORIZONTAL

SLIDER
-2
469
170
502
DECAY_COUNTER
DECAY_COUNTER
0
1
0.7
0.1
1
NIL
HORIZONTAL

SLIDER
-1
438
171
471
PANIC_DISCOUNT
PANIC_DISCOUNT
0
1
0.3
0.1
1
NIL
HORIZONTAL

TEXTBOX
5
195
93
308
Select ONE sociability distribution from the above toggles. Setting more than one to \"On\" will stop the model.
11
0.0
1

TEXTBOX
15
541
363
611
These sliders adjust model parameters. Use the \"Use_optimal_settings? toggle to set all settings to those used in Blanchard et al. 2021.
11
0.0
1

TEXTBOX
633
377
955
727
Grid or rand landscape toggle: On will use an evenly distributed gridded distribution of holes. Off will import the txt file containing a random hole distribution.\n\nShow social v asocial path: On will make all fish invisible except one random social and one random asocial fish, color them blue and red respectively, then ask them to put their pens down and track how far they've traveled. \n\nWatch a random fish: setting this to on will highlight one random fish, color their entire region of \"knowledge\" at the end of each day, open their inspection window, and provide a daily count of their observed local density in the command center.\n\nColor pool region by density is a button that sets the color of the region around solution holes to be colored relative to others. Warmer colors are higher densities.\n\nColor pool region by sociability is a button that sets the color of each solution hole as either white, if unoccupied, or on a blue scale relative their mean population sociability and prints the hole sociability in that space.
11
0.0
1

@#$#@#$#@
#See Supplementary Material 1 of Blanchard et al. 2021 for full ODD of this model.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.3.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="SETTINGS_OPTIMIZATION_EXP" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[density] of patches with [depth &lt; 0]</metric>
    <metric>standard-deviation [density] of patches with [depth &lt; 0]</metric>
    <metric>modes [density] of patches with [depth &lt; 0]</metric>
    <metric>median [density] of patches with [depth &lt; 0]</metric>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Big_search_radius" first="150" step="50" last="350"/>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="100"/>
      <value value="150"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Null?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Density_wiggle_room" first="0" step="0.25" last="0.5"/>
    <steppedValueSet variable="MAX_HANG_OUT_TIME" first="20" step="20" last="60"/>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="EXPERIMENT_SETTINGS?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="performance_evaluation" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[density] of patches with [depth &lt; 0]</metric>
    <metric>standard-deviation [density] of patches with [depth &lt; 0]</metric>
    <metric>modes [density] of patches with [depth &lt; 0]</metric>
    <metric>median [density] of patches with [depth &lt; 0]</metric>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor = 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor = 50]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor = 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor = 100]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor = 125]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor = 150]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor = 175]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor = 200]</metric>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Experiment_settings?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Social?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Null?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="patch color" repetitions="1" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[depth] of patches with [pycor &lt;= 20]</metric>
    <metric>[depth] of patches with [pycor &lt;= 40 AND pycor &gt; 20]</metric>
    <metric>[depth] of patches with [pycor &lt;= 60 AND pycor &gt; 40]</metric>
    <metric>[depth] of patches with [pycor &lt;= 80 AND pycor &gt; 60]</metric>
    <metric>[depth] of patches with [pycor &lt;= 100 AND pycor &gt; 80]</metric>
    <enumeratedValueSet variable="E_Social?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Experiment_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Null?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Asocial?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sensativity analysis for hang out time" repetitions="5" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 50]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 125]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 150]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Null?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <steppedValueSet variable="MAX_HANG_OUT_TIME" first="0" step="5" last="35"/>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Experiment_settings?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bimodal?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="density experiment grid" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt; 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 50]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 100]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 125]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 150]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 175]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Experiment_settings?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bimodal?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Null?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="density experiment rand" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt; 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 50]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 100]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 125]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 150]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 175]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Experiment_settings?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bimodal?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Null?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sociability experiment" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[SOCIABILITY_LIST] of patches with [depth &lt; 0]</metric>
    <metric>[DENSITY_LIST] of patches with [depth &lt; 0]</metric>
    <enumeratedValueSet variable="E_Social?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Null?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="60"/>
      <value value="120"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Experiment_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bimodal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Asocial?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sociability variance sdev density" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[DENSITY_LIST] of patches with [DEPTH &lt; 0]</metric>
    <metric>[SOCIABILITY_LIST] of patches with [DEPTH &lt; 0]</metric>
    <metric>[STDEV_LIST] of patches with [DEPTH &lt; 0]</metric>
    <metric>[VARIANCE_LIST] of patches with [DEPTH &lt; 0]</metric>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Null?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Social?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Asocial?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="60"/>
      <value value="120"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bimodal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Experiment_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="fixed sociability list output" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>[DENSITY_LIST] of patches with [DEPTH &lt; 0]</metric>
    <metric>[SOCIABILITY_LIST] of patches with [DEPTH &lt; 0]</metric>
    <metric>[STDEV_LIST] of patches with [DEPTH &lt; 0]</metric>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="60"/>
      <value value="120"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Null?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bimodal?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Social?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="E_Asocial?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Experiment_settings?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="POST MAP EXPERIMENT" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>DENSITY_LIST</metric>
    <metric>SOCIABILITY_LIST</metric>
    <metric>DISTANCE_LIST</metric>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Experiment_settings?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="200"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sensitivity analysis for wiggle room" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 26]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 51]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 76]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 101]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 126]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 151]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0"/>
      <value value="0.125"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sensitivity analysis for hang out time" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 26]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 51]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 76]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 101]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 126]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 151]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="0"/>
      <value value="2"/>
      <value value="5"/>
      <value value="10"/>
      <value value="20"/>
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sensitivity analysis for knowledge" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 50]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 125]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 150]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_SEARCH_RADIUS2">
      <value value="0.1"/>
      <value value="0.2"/>
      <value value="0.3"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sensitivity analysis for knowledge 2" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 50]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 125]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 150]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_SEARCH_RADIUS2">
      <value value="0.15"/>
      <value value="0.25"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sensitivity analysis for small angle" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 50]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 75]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 125]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 150]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="45"/>
      <value value="90"/>
      <value value="135"/>
      <value value="180"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sensitivity analysis fordispersal window" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 26]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 51]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 76]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 101]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 126]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 151]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="50"/>
      <value value="100"/>
      <value value="150"/>
      <value value="200"/>
      <value value="250"/>
      <value value="300"/>
      <value value="350"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sensitivity analysis for hangout decay_counter" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 26]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 51]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 76]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 101]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 126]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 151]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="DECAY_COUNTER">
      <value value="0.1"/>
      <value value="0.3"/>
      <value value="0.5"/>
      <value value="0.7"/>
      <value value="0.9"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="sensitivity analysis for panic discount" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 26]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 51]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 76]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 101]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 126]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 151]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="133"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="DECAY_COUNTER">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="PANIC_DISCOUNT">
      <value value="0.05"/>
      <value value="0.1"/>
      <value value="0.25"/>
      <value value="0.5"/>
      <value value="0.75"/>
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="manuscript_resubmission_rerun" repetitions="10000" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [linear_distance_traveled] of turtles</metric>
    <metric>mean [total_distance_traveled] of turtles</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 25]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 26]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 51]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 76]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 101]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 126]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 151]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="90"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="PANIC_DISCOUNT">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="75"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="DECAY_COUNTER">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_SEARCH_RADIUS2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="manuscript_resubmission_rerun_sociability_related" repetitions="10000" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>mean [TOTAL_DISTANCE_TRAVELED] of turtles</metric>
    <metric>SOCIABILITY_LIST</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 50 AND Pxcor &gt;= 26]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 75 AND Pxcor &gt;= 51]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 100 AND Pxcor &gt;= 76]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 125 AND Pxcor &gt;= 101]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 150 AND Pxcor &gt;= 126]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 175 AND Pxcor &gt;= 151]</metric>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &gt; 176]</metric>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="90"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="PANIC_DISCOUNT">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="DECAY_COUNTER">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_SEARCH_RADIUS2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="manuscript_resubmission_rerun_sociability_related" repetitions="1000" runMetricsEveryStep="false">
    <setup>setup</setup>
    <go>go</go>
    <metric>DISTANCE_LIST</metric>
    <metric>SOCIABILITY_LIST</metric>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="90"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="PANIC_DISCOUNT">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="20"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="DECAY_COUNTER">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_SEARCH_RADIUS2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="occupation analysis" repetitions="100" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>[density] of patches with [depth &lt; 0 AND Pxcor &lt;= 201]</metric>
    <enumeratedValueSet variable="MAX_HANG_OUT_TIME">
      <value value="3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Hetero?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Asocial?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Small_search_radius">
      <value value="90"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Control?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="PANIC_DISCOUNT">
      <value value="0.3"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dispersal_window">
      <value value="1"/>
      <value value="20"/>
      <value value="40"/>
      <value value="60"/>
      <value value="80"/>
      <value value="100"/>
      <value value="120"/>
      <value value="140"/>
      <value value="160"/>
      <value value="180"/>
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_target">
      <value value="40"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Show_social_v_asocial_path?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Max_moves_per_tick">
      <value value="30"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Use_optimal_settings?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Start_Fish_count">
      <value value="1000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="DECAY_COUNTER">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Grid_or_Rand_landscape?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="MAX_SEARCH_RADIUS2">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Watch_a_random_fish?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Big_search_radius">
      <value value="225"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Density_wiggle_room">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Social_rank_ordered_movement?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
