;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                                                              ;;
;;       Feedback Loop Example: Forest Resource Transport   December 2012       ;;
;;                                                                              ;;
;;  Code licenced by James D.A. Millington (http://www.landscapemodelling.net)  ;;
;;  under a Creative Commons Attribution-Noncommercial-Share Alike 3.0          ;;
;;  Unported License (see http://creativecommons.org/licenses/by-nc-sa/3.0/)    ;;
;;                                                                              ;;
;;  Model and documentation available from http://openabm.org                   ;;
;;                                                                              ;;
;;  Model used in:                                                              ;;
;;  Millington, J.D.A. (2013) Three types of spatial feedback loop in coupled   ;;
;;     human and natural systems. Ecology and Society [URL HERE]                ;;
;;                                                                              ;; 
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;



breed [ resources resource ] ;the resources being harvested
breed [ leaders leader ]     ;initial harvesters
breed [ followers follower ] ;harvesters that follow leaders

globals [ core-x core-y ]

patches-own [ friction ]

resources-own [ resource-level ] 

turtles-own 
[ 
  last-harvest ;location of where last harvest was made
  harvest-yield        ;yield of resources harvested
  led-by       ;leader being followed (for followers)
]


to setup
  
  ca
  random-seed number-seed
    
  ask patches [ set pcolor 59 ] ;all patches, will be background if not changed below
  
  set-default-shape turtles "circle"
  set core-x 0
  set core-y 0
  
  ask patches with [pxcor = 0 ] ; vertical roads always have no friction
  [ 
    set friction 0
    set pcolor 8
  ]
    
  ask patches with [pycor = 0 ] ; horizontal roads potentially have higher friction
  [ 
    if(road-friction = "None") [ set friction 0 ]
    if(road-friction = "Small-Diffc") [ set friction 1 ]
    if(road-friction = "Med-Diffc") [ set friction 2 ]
    if(road-friction = "Large-Diffc") [ set friction 3 ]
    set pcolor 8 - (friction * 2)
  ]
  
  ;create resources at top and sides of model environment
  ask patches with [(pycor = 0 or pxcor = 0) and (pycor = min-pycor or pycor = max-pycor or pxcor = max-pxcor or pxcor = min-pxcor)]
  [
    sprout-resources 1
    [
      set resource-level 100
      set shape "circle"
      set size (resource-level / 10)
    ]
  ]
  
  let current-colour 62
  ask resources
  [
    set color current-colour 
    set current-colour current-colour + 2
  ]    

  ;create harvesters at the centre of the model environment
  ask patch core-x core-y
  [
    set pcolor grey
    set friction 2
    
    sprout-leaders 4 
    [
      set size 1 
      set harvest-yield 0 
      set last-harvest nobody 
      set led-by self
    ]
    
    let current-heading 0 
    set current-colour 85
    ask leaders
    [
      set heading current-heading
      set color current-colour
      set current-heading current-heading + 90
      set current-colour current-colour + 10
      set size size * 2
    ]
     
    sprout-followers number-of-harvesters [set color blue set size 1 set harvest-yield 0 set last-harvest nobody set size size * 2]
  ]
  
  setup-plots 
  reset-ticks
    
end

to go
  
  if(not any? resources) [ stop ] ;once all resources have been harvestedm, stop simulation
  ask leaders
  [
    if(harvest-yield = 0 and last-harvest != nobody) [ face last-harvest ] ;all resource gone 
        
    move
    harvest
    deposit-harvest
  ]
  
  ask followers 
  [ 
    if(last-harvest != nobody) 
    [ 
      move
      harvest
      deposit-harvest
    ]
  ]
  
  if(any? followers with [ last-harvest = nobody]) [ wait 0.1 ] ;pause while followers choosing leaders to observe feedback easier
  if(pause-duration > 0) [ wait pause-duration ]  ;pause to view transport easier if desired
   
  ask resources [ set size resource-level / 10 ]
  
  tick
  
end


to move
  
  ;movement of harvesters is modified by the friction of the patch they travel across
  ;if at the core, add a random delay so that harvesters separate out (so we can see them)
  ifelse([pxcor] of patch-here = core-x and [pycor] of patch-here = core-y) 
  [
    let f random 2
    if(f = 1) [ fd 5 - [friction] of patch-here ]
  ]
  [ fd 5 - [friction] of patch-here ]
  
end

  
  
to harvest
  
  ;collect material from resource and turn around
  if(any? resources-here) 
  [
    set harvest-yield harvest-capacity
    ask resources-here [ set resource-level resource-level - harvest-capacity ]
    set last-harvest patch-here
    face patch core-x core-y
    
    ask resources-here 
    [
      if(resource-level <= 0) [ die ]
    ]
  ]

end
  
  
to deposit-harvest

   ;if harvester holds material and is back at the core area, release this material and turn to face last harvest location
   if(harvest-yield > 0 and [pxcor] of patch-here = core-x and [pycor] of patch-here = core-y)
   [
     set harvest-yield 0
     face last-harvest
     
     ;if there are harvesters at the core not currently following another harvester, tell one of them to harvest from the location I am harvesting at
     if(any? followers-here with [last-harvest = nobody])
     [
       ask one-of followers-here with [last-harvest = nobody]
       [
         set last-harvest [last-harvest] of myself
         set led-by myself 
         set color [color] of myself
         face last-harvest
       ]
     ]
   ]
   
end

    
@#$#@#$#@
GRAPHICS-WINDOW
230
10
589
390
40
40
4.31
1
10
1
1
1
0
0
0
1
-40
40
-40
40
0
0
1
ticks
30.0

BUTTON
7
15
74
48
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
80
15
143
48
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
149
15
212
48
step
go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
9
67
209
217
Harvesters
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"ask leader 4 [ set-current-plot-pen \\"pen-1\\" set-plot-pen-color color]\\nask leader 5 [ set-current-plot-pen \\"pen-2\\" set-plot-pen-color color]\\nask leader 6 [ set-current-plot-pen \\"pen-3\\" set-plot-pen-color color]\\nask leader 7 [ set-current-plot-pen \\"pen-4\\" set-plot-pen-color color]" "ask leader 4 [ set-current-plot-pen \\"pen-1\\" plotxy ticks (count turtles with [led-by = leader 4]) ]\\nask leader 5 [ set-current-plot-pen \\"pen-2\\" plotxy ticks (count turtles with [led-by = leader 5]) ]\\nask leader 6 [ set-current-plot-pen \\"pen-3\\" plotxy ticks (count turtles with [led-by = leader 6]) ]\\nask leader 7 [ set-current-plot-pen \\"pen-4\\" plotxy ticks (count turtles with [led-by = leader 7]) ]"
PENS
"pen-1" 1.0 0 -16777216 true "" ""
"pen-2" 1.0 0 -7500403 true "" ""
"pen-3" 1.0 0 -2674135 true "" ""
"pen-4" 1.0 0 -955883 true "" ""

PLOT
8
240
208
390
Resources
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"ask resource 0 [ set-current-plot-pen \\"pen-1\\" set-plot-pen-color color]\\nask resource 1 [ set-current-plot-pen \\"pen-2\\" set-plot-pen-color color]\\nask resource 2 [ set-current-plot-pen \\"pen-3\\" set-plot-pen-color color]\\nask resource 3 [ set-current-plot-pen \\"pen-4\\" set-plot-pen-color color]" "if(is-turtle? resource 0) [ask resource 0 [ set-current-plot-pen \\"pen-1\\" plotxy ticks resource-level ]]\\nif(is-turtle? resource 1) [ask resource 1 [ set-current-plot-pen \\"pen-2\\" plotxy ticks resource-level ]]\\nif(is-turtle? resource 2) [ask resource 2 [ set-current-plot-pen \\"pen-3\\" plotxy ticks resource-level ]]\\nif(is-turtle? resource 3) [ask resource 3 [ set-current-plot-pen \\"pen-4\\" plotxy ticks resource-level ]]"
PENS
"pen-4" 1.0 0 -16777216 true "" ""
"pen-1" 1.0 0 -7500403 true "" ""
"pen-2" 1.0 0 -2674135 true "" ""
"pen-3" 1.0 0 -955883 true "" ""

SLIDER
24
407
196
440
number-seed
number-seed
0
1000
739
1
1
NIL
HORIZONTAL

SLIDER
23
446
195
479
pause-duration
pause-duration
0
0.2
0
0.05
1
s
HORIZONTAL

CHOOSER
231
433
369
478
road-friction
road-friction
"None" "Small-Diffc" "Med-Diffc" "Large-Diffc"
3

SLIDER
394
406
584
439
number-of-harvesters
number-of-harvesters
0
100
50
1
1
NIL
HORIZONTAL

SLIDER
394
444
586
477
harvest-capacity
harvest-capacity
0.2
1
1
0.2
1
NIL
HORIZONTAL

TEXTBOX
236
398
386
430
Darker road colour indicates more friction
12
0.0
1

@#$#@#$#@
## WHAT IS IT?

This model illustrates a positive 'transport' feedback loop described in Millington (2013). In this type of feedback a line of reduced or increased resistance to flows of information and/or material results in increased or decreased rates of change in linked entities. This particular example represents the role of roads in tropical deforestation in the Amazon basin (e.g., Malanson et al. 2006), facilitating the flow of people and material from settled (core) regions to frontier (periphery) regions, and vice versa. A positive feedback loop is formed as greater flows of people and materials improve economic returns on activities at the periphery, which in turn stimulate improved and greater flows of materials and people along the roads to core markets and elsewhere. Large green circles indicate resources (e.g., forest), smaller dots indicate resource harvesters, grey lines are roads (darker, lower transport cost). 

## HOW IT WORKS

At initialization, the model environment contains four resources at the periphery each linked by a road to the core area at the centre of the model environment. Harvesters originate from the core area. Initially there are only four harvesters, one for each of the resources. 

In each timestep harvesters move along roads towards either resource or core. Their speed of movement is restricted by the friction of the road (reflecting road quality and/or length). On arriving at a resource harvesters collect material from the resource, reducing its resource-level. Harvesters then return to the core area to deposit their harvest before returning again to the same resource for further harvest. 

While at the core area, if a harvester meets another harvester who has not yet begun to harvest, the second harvester follows the first harvester to the resource. In this way a feedback is created, as roads with lower friction result in more harvesters using them because those using these roads visit the core (collecting new harvesters) more frequently. 

## HOW TO USE IT

First, decide what road friction levels you want to examine by selecting a _road-friction_ choice:

1. **None:** no difference between vertical and horizontal roads (both have friction = 0)
2. **Small-Diffc:** a small difference between vertical and horizontal roads (i.e., vertical have friction = 0 and horizontal have friction = 1)
3. **Med-Diffc:** an intermediate difference between vertical and horizontal roads (i.e., vertical have friction = 0 and horizontal have friction = 2)
4. **Large-Diffc:** a large difference between vertical and horizontal roads (i.e, vertical have friction = 0 and horizontal have friction = 3)

Next, decide how many harvesters should be simulated (by moving the _number-of-harvesters_ slider) and how much resource they can harvest in each visit to a resource (by moving the _harvest-capacity_ slider). 

To change the randomization of the simulation you can move the _number-seed_ slider. 

The simulation automatically speeds up once all harvesters have decided which resource they will harvest from. To slow subsequent simulation increase _pause-duration_.

## THINGS TO NOTICE

Notice how roads with lower friction end up with more harvesters using them and consequently how resources connected by those roads to the core are depleted faster. The upper plot shows the number of harvesters in each of the colours through time and the lower plot shows the resources remaining in each resource (by colour).

## THINGS TO TRY

Experiment with how different combinations of number of harvesters, harvest capacity and road friction result in different rates of depletion between resources.

## CREDITS AND REFERENCES
Malanson, G. P., Zeng, Y., & Walsh, S. J. (2006). Landscape frontiers, geography frontiers: lessons to be learned. The Professional Geographer, 58(4), 383-396

Millington, J.D.A. (2013) Three types of spatial feedback loop in coupled human and natural systems. Ecology and Society [URL HERE]                

Code licenced by James D.A. Millington (http://www.landscapemodelling.net) under a Creative Commons Attribution-Noncommercial-Share Alike 3.0 Unported License (see http://creativecommons.org/licenses/by-nc-sa/3.0/)                                                                       
Model and documentation available from http://openabm.org   
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
0
Rectangle -7500403 true true 151 225 180 285
Rectangle -7500403 true true 47 225 75 285
Rectangle -7500403 true true 15 75 210 225
Circle -7500403 true true 135 75 150
Circle -16777216 true false 165 76 116

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -7500403 true true 135 285 195 285 270 90 30 90 105 285
Polygon -7500403 true true 270 90 225 15 180 90
Polygon -7500403 true true 30 90 75 15 120 90
Circle -1 true false 183 138 24
Circle -1 true false 93 138 24

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
