;Dr. Lin Xiang created this module at the University of Kentucky in 2022. If you mention this model in a publication, we ask that you include the citations below.
;
;Xiang, L. (2022). Dissolved Inorganic Carbon in a closed system. Department of STEM Education, University of Kentucky, Lexington, KY.
;
;![CC BY-NC-SA 4.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)
;
;This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 4.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/4.0/.



breed [H2CO3s H2CO3]
breed [HCO3s HCO3]
breed [CO3s CO3]
breed [H3OS H30]
breed [OHs OH]
breed [legends legend]

turtles-own[ ]
globals [
  Num-H
  Num-OH
  Num-H2CO3
  Num-HCO3
  Num-CO3
  current-ph
  current-pOH
  current-conc-H
  current-conc-OH
  current-conc-H2CO3s
  current-conc-HCO3s
  current-conc-CO3s
  Ka1
  Ka2
  TC
  denom
  num
  ]

to setup
  ca
  set-bk
  set-legends
  set-shapes
  set-tt
  plot-ion-concentrations-log
  reset-ticks
end

to set-bk
  ask patches [set pcolor 96.5]
  ask patches with [ pycor = max-pycor or pycor < 6] [set pcolor 101]
  ask patches with [ pxcor = min-pxcor or pxcor = max-pxcor] [set pcolor 101]
end


to set-tt

  ;set constants
  set TC 0.002
  set Ka1 10 ^ (-5.903)         ;from Millero et al. (2006)
  set Ka2 10 ^ (-9.702)         ;;from Millero et al. (2006)

  ;set initial carbon species
  set ph 0
  set-carbon-concentrations

  ; set particles
  set num-H2CO3 round (5000 * 3 ^ (log current-conc-H2CO3s 10))
  set num-HCO3 round (5000 * 3 ^ (log current-conc-HCO3s 10))
  set num-CO3 round (5000 * 3 ^ (log current-conc-CO3s 10))

  create-H2CO3s Num-H2CO3  [set-tt-properties-aq]
  create-HCO3s Num-HCO3  [set-tt-properties-aq]
  create-CO3s Num-CO3  [set-tt-properties-aq]

  set num-H round (500 * 1.3895 ^ (-1 * current-ph))
  set num-OH round (500 * 1.3895 ^ (-1 * (14 - current-ph)))

  create-h3os num-H [set-tt-properties-aq ]
  create-ohs num-OH [set-tt-properties-aq ]

  ;hiding ions
  if Hide-H2CO3? = true
    [if any? H2CO3s [ask H2CO3s [die]]]

  if hide-HCO3? = true
    [if any? HCO3s [ask HCO3s [die]]]

  if hide-CO3? = true
    [if any? CO3s [ask CO3s [die]]]

  if Hide-H3Os-and-OHs? = true
    [if any? H3Os [ask H3Os [die]]
     if any? OHs [ask OHs [die]]]
 end

to set-carbon-concentrations

  set current-ph pH
  set current-conc-H 10 ^ (- current-ph)

  set denom (current-conc-H * current-conc-H + Ka1 * Ka2 + current-conc-H * Ka1)
  set current-conc-H2CO3s (current-conc-H * current-conc-H * TC) / denom
  set current-conc-HCO3s (Ka1 * current-conc-H * TC) / denom
  set current-conc-CO3s (Ka1 * Ka2 * TC) / denom
end


to set-tt-properties-aq
  setxy 1 + random-float (max-pxcor - 2) 6 + random-float (max-pycor - 7)
  (ifelse
    choose-a-visual = "Molecule visual" [set size 1.25]
    choose-a-visual = "Icon visual" [set size 1.75])
end

to set-legends
 create-legends 10 [set label-color white]

 ask legend 0[setxy 2 3.5]

 ask legend 1
   [set shape "blank" set size 1
    set label "Hydroxide(OH-)" setxy 10 3.9]

  ask legend 2 [setxy 2 1]

  ask legend 3
  [set shape "blank" set size 1
    set label "Hydronium (H3O+)" setxy 11 1.3]

  ask legend 4[setxy 28 3.5]

  ask legend 5
  [set shape "blank" set size 1.5
    set label "Carbonate (CO3--)" setxy 37.4 3.9]

  ask legend 6 [setxy 14 3.5]

  ask legend 7
  [set shape "blank" set size 1.5
    set label "Carbonic acid (H2CO3)" setxy 25 3.9]

   ask legend 8 [setxy 14 1]

  ask legend 9
  [set shape "blank" set size 1.5
    set label "Bicarbonate (HCO3-)" setxy 24 1.3]

end


to go

set-shapes-1
aq-move
set-carbon-concentrations
display-particles
plot-ion-concentrations-log

tick
end

to display-particles

  set num-H2CO3 round (5000 * 3 ^ (log current-conc-H2CO3s 10))
  set num-HCO3 round (5000 * 3 ^ (log current-conc-HCO3s 10))
  set num-CO3 round (5000 * 3 ^ (log current-conc-CO3s 10))


ifelse Hide-H2CO3? = false
  [
   let current-H2CO3 count H2CO3s

  (ifelse
    current-H2CO3 > num-H2CO3
     [ask up-to-n-of (current-H2CO3 - num-H2CO3) H2CO3s [die]]
    current-H2CO3 < num-H2CO3
     [create-H2CO3s (num-H2CO3 - current-H2CO3) [set-tt-properties-aq]])
    ]

  [if any? H2CO3s [ask H2CO3s [die]]]


ifelse hide-HCO3? = false
    [let current-HCO3 count HCO3s
  (ifelse
    current-HCO3 > num-HCO3
     [ask up-to-n-of (current-HCO3 - num-HCO3) HCO3s [die]]
    current-HCO3 < num-HCO3
     [create-HCO3s (num-HCO3 - current-HCO3) [set-tt-properties-aq]])]
    [if any? HCO3s [ask HCO3s [die]]]

ifelse hide-CO3? = false
    [let current-CO3 count CO3s
  (ifelse
    current-CO3 > num-CO3
     [ask up-to-n-of (current-CO3 - num-CO3) CO3s [die]]
    current-CO3 < num-CO3
     [create-CO3s (num-CO3 - current-CO3) [ set-tt-properties-aq]])]
    [if any? CO3s [ask CO3s [die]]]

ifelse Hide-H3Os-and-OHs? = false
    [ H-OH-equilibrium]
    [if any? H3Os [ask H3Os [die]]
     if any? OHs [ask OHs [die]]]
end

to H-OH-equilibrium

  set num-H round (500 * 1.3895 ^ (-1 * current-ph))
  set num-OH round (500 * 1.3895 ^ (-1 * (14 - current-ph)))

  let current-H count H3Os
  let current-OH count OHs

  (ifelse
    num-H > current-H
     [create-h3os abs (num-H - current-H) [set-tt-properties-aq ]]
    num-H < current-H
    [ask n-of abs (num-H - current-H) h3os [die]])

   (ifelse
    num-OH > current-OH
    [create-ohs abs (num-OH - current-OH) [set-tt-properties-aq ]]
    num-OH < current-OH
    [ask n-of abs (num-OH - current-OH) OHs [die]])

end

to aq-move
  ask turtles with [ breed != legends]    ;ask aq turtles
     [(ifelse
        [pcolor] of patch-at dx dy = 101 or [pcolor] of patch-at dx dy = 89  [rt random 360]    ;if it's boundary ahead, rotate
        [pcolor] of patch-at dx dy = 96.5  [fd 0.2 rt random 360 if ycor < 6 [set ycor 6]] ;if the patch-ahead has the proper color, move forward.If ycor is low, put ycor higher
          )]
end

to plot-ion-concentrations-log ;; this creates the bar graph

    set-current-plot "Relative Concentrations"
    clear-plot
    set-plot-x-range 0 2
    set-plot-y-range 0 300
    plot-pen-down
    set-current-plot-pen "H2CO3"
    plotxy 0 count H2CO3s
    set-current-plot-pen "HCO3-"
    plotxy 1 count HCO3s
    set-current-plot-pen "CO3(2-)"
    plotxy 2 count CO3s


end

to set-shapes
  (ifelse
    choose-a-visual = "Molecule visual"
    [set-default-shape ohs "ab-oh-1"
     set-default-shape H3Os "ab-h3o-1"
     set-default-shape H2CO3s "ab-H2CO3"
     set-default-shape HCO3s "ab-HCO3"
     set-default-shape CO3s "ab-CO3"
      ask legend 0 [set shape "ab-oh-1" set size 1.75]
      ask legend 2 [set shape "ab-h3o-1" set size 1.75]
      ask legend 6 [set shape "ab-H2CO3" set size 1.75]
      ask legend 8 [set shape "ab-HCO3" set size 1.75]
      ask legend 4 [set shape "ab-CO3" set size 1.75]
    ]

   choose-a-visual = "Icon visual"
    [set-default-shape ohs "ab-oh-2"
     set-default-shape H3Os "ab-h3o-2"
     set-default-shape H2CO3s "ab-H2CO3-1"
     set-default-shape HCO3s "ab-HCO3-1"
     set-default-shape CO3s "ab-CO3-1"
      ask ohs [set shape "ab-oh-2" set size 1.25]
      ask H3Os [set shape "ab-h3o-2" set size 1.25]
      ask legend 0 [set shape "ab-oh-2" set size 1.75]
      ask legend 2 [set shape "ab-h3o-2" set size 1.75]
      ask legend 6 [set shape "ab-H2CO3-1" set size 1.75]
      ask legend 8 [set shape "ab-HCO3-1" set size 1.75]
      ask legend 4 [set shape "ab-CO3-1" set size 1.75]
  ])
end

to set-shapes-1
  (ifelse
    choose-a-visual = "Molecule visual"
    [set-default-shape ohs "ab-oh-1"
     set-default-shape H3Os "ab-h3o-1"
     set-default-shape H2CO3s "ab-H2CO3"
     set-default-shape HCO3s "ab-HCO3"
     set-default-shape CO3s "ab-CO3"
     ask ohs [set shape "ab-oh-1" set size 1.25]
     ask H3Os [set shape "ab-h3o-1" set size 1.25]
     ask H2CO3s [set shape "ab-H2CO3" set size 1.25]
     ask HCO3s [set shape "ab-HCO3" set size 1.25]
     ask CO3s [set shape "ab-CO3" set size 1.25]
      ask legend 0 [set shape "ab-oh-1" set size 1.75]
      ask legend 2 [set shape "ab-h3o-1" set size 1.75]
      ask legend 6 [set shape "ab-H2CO3" set size 1.75]
      ask legend 8 [set shape "ab-HCO3" set size 1.75]
      ask legend 4 [set shape "ab-CO3" set size 1.75]
    ]

   choose-a-visual = "Icon visual"
    [set-default-shape ohs "ab-oh-2"
     set-default-shape H3Os "ab-h3o-2"
     set-default-shape H2CO3s "ab-H2CO3-1"
     set-default-shape HCO3s "ab-HCO3-1"
     set-default-shape CO3s "ab-CO3-1"
     ask ohs [set shape "ab-oh-2" set size 1.75]
     ask H3Os [set shape "ab-h3o-2" set size 1.75]
     ask H2CO3s [set shape "ab-H2CO3-1" set size 1.75]
     ask HCO3s [set shape "ab-HCO3-1" set size 1.75]
     ask CO3s [set shape "ab-CO3-1" set size 1.75]
      ask legend 0 [set shape "ab-oh-2" set size 1.75]
      ask legend 2 [set shape "ab-h3o-2" set size 1.75]
      ask legend 6 [set shape "ab-H2CO3-1" set size 1.75]
      ask legend 8 [set shape "ab-HCO3-1" set size 1.75]
      ask legend 4 [set shape "ab-CO3-1" set size 1.75]
  ])
end
@#$#@#$#@
GRAPHICS-WINDOW
7
99
466
471
-1
-1
11.0
1
10
1
1
1
0
1
1
1
0
40
0
32
1
1
1
ticks
10.0

BUTTON
138
10
254
56
Set up/Restart
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
7
60
134
93
Run/Pause
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

PLOT
470
99
866
347
Carbonate Percentages
pH
Percentage
0.0
14.0
0.0
100.0
true
true
"" ""
PENS
"H2CO3" 1.0 0 -4079321 true "" "plotxy ph (current-conc-H2CO3s * 100 / TC)"
"HCO3-" 1.0 0 -8630108 true "" "plotxy ph (current-conc-HCO3s * 100 / TC)"
"CO3(2-)" 1.0 0 -10899396 true "" "plotxy ph (current-conc-CO3s * 100 / TC)"

PLOT
470
348
866
574
Relative Concentrations
Ion
Relative Concentration
0.0
2.0
0.0
300.0
true
true
"" ""
PENS
"H2CO3" 1.0 1 -4079321 true "" "plotxy 0 count h2co3s"
"HCO3-" 1.0 1 -8630108 true "" "plotxy 1 count hco3s"
"CO3(2-)" 1.0 1 -10899396 true "" "plotxy 2 count co3s"

MONITOR
7
491
154
532
H2CO3 (Mol/L)
current-conc-H2CO3s
17
1
10

SLIDER
257
60
540
93
pH
pH
0
precision 14 1
0.0
0.1
1
NIL
HORIZONTAL

MONITOR
155
491
310
532
HCO3- (Mol/L)
current-conc-HCO3s
17
1
10

MONITOR
311
491
466
532
CO3(2-) (Mol/L)
current-conc-CO3s
17
1
10

MONITOR
155
533
310
574
H3O+ (Mol/L)
current-conc-H
14
1
10

MONITOR
311
533
466
574
OH- (Mol/L)
10 ^ (-14) / current-conc-H
14
1
10

SWITCH
257
18
398
51
Hide-H2CO3?
Hide-H2CO3?
1
1
-1000

SWITCH
400
18
540
51
Hide-HCO3?
Hide-HCO3?
1
1
-1000

SWITCH
542
18
653
51
Hide-CO3?
Hide-CO3?
1
1
-1000

SWITCH
655
18
825
51
Hide-H3Os-and-OHs?
Hide-H3Os-and-OHs?
1
1
-1000

TEXTBOX
548
58
876
92
1: Choose a visual style. Set up model and run model. \n2. Change pH and observe results, or auto plot the results.\n    (Optional): Hide some ions to help observations.
9
125.0
1

CHOOSER
7
10
135
55
Choose-a-visual
Choose-a-visual
"Molecule visual" "Icon visual"
1

TEXTBOX
8
474
158
492
Ion Concentration:
11
105.0
1

BUTTON
138
60
254
93
Auto Plot
ifelse ph < 13.9\n[go set ph ph + 0.1][stop]
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
0

@#$#@#$#@
## WHAT IS IT?

Carbonic acid (H2CO3), Bicarbonate ion (HCO3-), and carbonate ion (CO3 (2-))are three species of dissolved inorganic carbon (DIC) that coexist in water. In a closed system, assuming that no carbon leaves or enters the system, the proportions of these species are determined by many factors. This model presents the proportions of three DIC species at different pH levels, ranging from 0 to 14, holding other factors constant. This model may serve as the foundation for students to explore the phenomenon of ocean acidification. 

## HOW IT WORKS

In the model, a total DIC concentration of 0.002 (mol kg-1) is used. The values of two equilibrium constants, Ka1 = 10 ^ (-5.903) and Ka2 = 10 ^ (-9.702), obtained from Millero et al. (2006), are used to calculate the concentrations of three species in a closed system where salinity is 3.3758% and temperature is 20 degree celsius.

The following equations are used in calculations:

1)  [H2CO3]+[HCO3-]+[CO3--]= Total Carbon (TC)
2)  [HCO3-][H+]/[H2CO3]=Ka1
3)  [CO3--][H+]/[HCO3]=Ka2

Equations 2-3 can be rearranged:
4)  [H2CO3]=[HCO3-][H+]/Ka1
5)  [CO3--]=Ka2*[HCO3]/[H+]

Equations 4-5 can be substituted in equation 1 to solve [HCO3-]:

6)  [HCO3-] = (TC * Ka1 * [H+])/([H+]^2 + Ka1 * [H+] + Ka1 * Ka2)

Then [HCO3-] can be substituted in equations 4-5 to solve corresponding [H2CO3] and [CO3--] at the same pH level. 

The percentage of each species is calculated using equation 7:

7)  Percentage of [species] = [species] * 100 / TC

## HOW TO USE IT

1. Choose a visual style. The icon visual is more visually friendly, and the molecule visual provides more authentic molecular properties of the species. 

2. Click on the **set up/restart** button the confirm the setting.

3. You may **run** the model and then drag the pH slider to observe how the proportions of three species vary at the pH ranging from 0 to 14. 

4. You may also let the model ** auto plot** the changes for you. 

## THINGS TO TRY AND NOTICE

The main observation is to see how the concentration of three species changes at different ph levels by dragging the pH slider or using the auto plot button. The teacher may also guide students to see that the current ph change in the ocean concerns a small range (8-8.2) in the spectrum.  

Use switches to hide some ions to focus your observation. For example, only show H2CO3 and hide the rest ions. 


## RELATED MODELS

See more chemistry models at https://sites.google.com/view/3d-science-abm/physics-chemistry

## CREDITS AND REFERENCES

Dr. Lin Xiang created this module at the University of Kentucky in 2022. If you mention this model in a publication, we ask that you include the citations below.

Xiang, L. (2022). Dissolved Inorganic Carbon in a closed system. Department of STEM Education, University of Kentucky, Lexington, KY.

![CC BY-NC-SA 4.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 4.0 License.  To view a copy of this license, visit https://creativecommons.org/licenses/by-nc-sa/4.0/.

References:
Millero, F. J.; Graham, T. B.; Huang, F.; Bustos-Serrano, H.; Pierrot, D. Dissociation constants of carbonic acid as a function of salinity and temperature. (2006). Marine Chemistry, 100 (1-2), 80-94.

This model is inspired by the KCVS applets from the 2100 C.E. RCP projections. https://applets.kcvs.ca/OceanAcidification/oceanAcid.html?rcp=2.6 
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

ab-co2
true
0
Circle -2674135 true false 180 90 120
Circle -16777216 true false 75 75 150
Circle -2674135 true false 0 90 120

ab-co2-1
true
0
Circle -2064490 true false 90 90 120

ab-co3
true
0
Circle -2674135 true false 120 0 120
Circle -16777216 true false 60 60 180
Circle -2674135 true false 0 90 120
Circle -2674135 true false 150 165 120

ab-co3-1
true
0
Polygon -1 true false 150 30 30 225 270 225

ab-h2co3
true
0
Circle -2674135 true false 120 0 120
Circle -16777216 true false 60 60 180
Circle -2674135 true false 0 90 120
Circle -2674135 true false 150 165 120
Circle -1 true false 122 0 42
Circle -1 true false 163 253 42

ab-h2co3-1
true
0
Polygon -1184463 true false 150 30 30 225 270 225
Circle -955883 true false 116 11 67
Circle -955883 true false 11 176 67

ab-h2o-1
true
0
Circle -7500403 false true 75 75 148
Circle -7500403 true true 84 189 42
Circle -7500403 true true 204 144 42

ab-h3o-1
true
0
Circle -1 true false 114 54 42
Circle -2674135 true false 75 75 150
Circle -1 true false 69 189 42
Circle -1 true false 204 159 42

ab-h3o-2
true
0
Polygon -2674135 true false 120 75 195 75 225 135 195 195 120 195 90 135

ab-hco3
true
0
Circle -2674135 true false 120 0 120
Circle -16777216 true false 60 60 180
Circle -2674135 true false 0 90 120
Circle -2674135 true false 150 165 120
Circle -1 true false 163 253 42

ab-hco3-1
true
0
Polygon -8630108 true false 150 30 30 225 270 225
Circle -955883 true false 116 11 67

ab-oh-1
true
0
Circle -13345367 true false 75 75 150
Circle -1 true false 189 174 42

ab-oh-2
true
0
Polygon -13345367 true false 120 75 195 75 225 135 195 195 120 195 90 135

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

blank
true
0

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

factory
false
0
Rectangle -7500403 true true 76 194 285 270
Rectangle -7500403 true true 36 95 59 231
Rectangle -16777216 true false 90 210 270 240
Line -7500403 true 90 195 90 255
Line -7500403 true 120 195 120 255
Line -7500403 true 150 195 150 240
Line -7500403 true 180 195 180 255
Line -7500403 true 210 210 210 240
Line -7500403 true 240 210 240 240
Line -7500403 true 90 225 270 225
Circle -1 true false 37 73 32
Circle -1 true false 55 38 54
Circle -1 true false 96 21 42
Circle -1 true false 105 40 32
Circle -1 true false 129 19 42
Rectangle -7500403 true true 14 228 78 270

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.2.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="50"/>
    <metric>current-ph</metric>
    <steppedValueSet variable="P-CO2" first="255" step="5" last="900"/>
    <enumeratedValueSet variable="ph">
      <value value="7"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
