;; Monday 7th Jan 2013
;; Ball bouncing around in a box as a represtentation of a hazard
;; agents have individual resilience - time taken to recover and susceptibility
;; adaptive capacity - capablility to build barriers that can shelter from or deflect the ball
;; cooperation of community to build shared protection, recovery, hazard perception

;;extensions [profiler]
breed [balls ball]
breed [people person]
breed [barriers barrier]
people-own [ damage recovery susceptibility latest total total-here record cbas recent relocs] 
barriers-own [owner end-of-life times causals] 
balls-own [route]
globals [
  number-of-balls number-of-people 
  number-of-collisions 
  habitable outer 
  build-barriers
  relocation-rule
  barrier-lifespan
  mrad
  max-recovery
  min-recovery
  network-type
]

to setup-balls
  clear-all
  set number-of-balls 2
  set barrier-lifespan lifespan?
  if habitable? = "single" [  set habitable patches with [abs pxcor < 7 and abs pycor < 7]]
  if habitable? = "double" [  
    let lside patches with [abs pxcor < 7 and pxcor < 0 and abs pycor < 7]
    let rside patches with [abs pxcor < 7 and pxcor > 0 and abs pycor < 7] 
    set habitable (patch-set lside rside)
  ]
  if habitable? = "twin" [
    let lside patches with [abs pxcor < 9 and pxcor < 0 and abs pycor < 9 and pycor < 0]
    let rside patches with [abs pxcor < 9 and pxcor > 0 and abs pycor < 9 and pycor > 0]
    set habitable (patch-set lside rside)
  ]
   if habitable? = "four" [
    let b-l patches with [abs pxcor < 9 and pxcor < -2 and abs pycor < 9 and pycor < -2]
    let t-l patches with [abs pxcor < 9 and pxcor < -2 and abs pycor < 9 and pycor > 2]
    let t-r patches with [abs pxcor < 9 and pxcor > 2 and abs pycor < 9 and pycor > 2]
    let b-r patches with [abs pxcor < 9 and pxcor > 2 and abs pycor < 9 and pycor < -2]
    set habitable (patch-set b-l t-l t-r b-r)
  ]
  ask habitable [set pcolor 3]  ;; dark grey
  set outer patch-set filter [not member? ? habitable] sort patches
  ask outer [set pcolor black]  ;; dark grey
  print (word "set up habitable zone: " habitable?)
  ;; make green balls
  set-default-shape balls "circle"
  create-balls number-of-balls
    [ set color green
      setxy random-xcor random-ycor
      set route []
    ]
  ;; create permanent barriers - set end-of-life 0 
  set-default-shape barriers "barrier"
  create-barriers 8
  [
    set color 34 ;; use colours 34 to 39 (brown to white/black)
    move-to one-of outer with [not any? turtles-here]
    set times 0
    set causals 0
    set end-of-life 0
  ]  
  reset-ticks
end

to setup
  setup-balls
  
  set build-barriers build-barriers?   ;; variables can be set or chosen via interface
  set relocation-rule relocation-rule?
  set network-type network-type?
  
  set number-of-people number-of-people?
  set number-of-collisions 0
  set mrad 3                           ;; farthest distance which people can "see" to monitor 
  set mrad chosen-mrad
  set max-recovery 50
  set min-recovery 10
  set max-recovery chosen-max-recovery
  set min-recovery chosen-min-recovery
  
  set-default-shape people "person"
  create-people number-of-people
    [ set color brown
      move-to one-of habitable with [not any? turtles-here]
      ;;setxy random-pxcor random-pycor                    ;; move to the centre of a random patch
      set damage 0                                         ;; damage to person increases with each incident 
      set recovery one-of (list min-recovery max-recovery) ;; time taken until recovery (with no intervention)
      set susceptibility random 2                          ;; 1 - susceptible / 0 - not susceptible
      set susceptibility 1     
      set latest 0                     ;; time of last occurance of damage
      set total 0                      ;; total damaging events
      set record []                    ;; record of the monitoring of a ball
      set cbas []                      ;; causal (links) from barriers to actors
      set total-here 0                 ;; damages received at the current location
      set relocs 0                     ;; number of times the agent has relocated
    ]
  create-network
end

;; local network shows links with nearest neighbours (which need to be reset if relocate strategy is used)
;; random network can include many connections to distant agents
to create-network
  ifelse network-type = "Off" 
  [ print (word "No network created ")]
  [ print (word "created " network-type " network")]
  if network-type = "local" [
    ;; The agentset is built by finding all the agents with the highest value of reporter,
    ask people [create-links-with min-n-of 2 other people [distance myself]]
  ]
  if network-type = "random" [
    ask people [ create-links-with n-of 2 other people ]
  ]
end

;; perturbation coming from outer edge heading towards habitable
;; does not destroy original barriers but will destroy constructed ones
to perturbate
  let chosen-ball one-of balls 
  print (word "peturbation will be caused by " chosen-ball)
  ask chosen-ball [
    let target one-of habitable
    let rnos (list random 2 random 2)
    ifelse item 0 rnos = 0 [
      set xcor [pxcor] of target 
      ifelse item 1 rnos = 1 [set ycor max-pycor ][set ycor min-pycor]
    ]
    [
      set ycor [pycor] of target
      ifelse item 1 rnos = 1 [set xcor max-pxcor ][set xcor min-pxcor]
    ]
    set heading towards target
    set size (size + 1)  
  ]
  ;; repeat for sufficient ticks to reach other edge of world
  repeat (2 * max-pxcor) [go]
  ask chosen-ball [
    move-to one-of outer
    set size (size - 1) 
    set heading random 360
  ] 
end

;; if the ball is stuck start it again from outside of habitable
to dislodge
  ;; if the route caused a lot of collsions or TODO if there is a repeating pattern??
  ask balls [
    if length route > 10 [
      move-to one-of outer
      print word "dislodged " self
      set route []
    ]
    set route []  ;; reset for all cases   
  ]  
end

;; clear barriers causing damages above a tolerance level, plus any on neighbours4 patches with same heading
to clear-barriers
  ;; check all barriers, obtain subset of damaging ones
  let adbs filter [[causals]  of ? > 0] sort barriers
  foreach adbs [
    if ? != nobody [
      if [causals] of ? > (tolerance * [times] of ?) 
      [
        let this-heading [heading] of ?
        ;; remove barriers on n4 patches with same heading 
        ask (barriers-on [neighbors4] of ?) with [heading = this-heading] [die]
        ;; then remove this barrier
        ask ? [die]  
      ] 
    ]
  ]
  ;; also, periodically, remove barriers in habitable at end-of-life
  ask barriers with [end-of-life < ticks and end-of-life > 0] [die]
end

to go
  ;; periodic clean up, followed by mitigation build stage and relocate stage
  if ticks mod 50 = 0 [
    clear-barriers
    dislodge
    ask people [ if damage = 0 [ mitigate ]]
  ]
  ask people [
    ;; mitigation and recovery as two distinct processes
    if damage > 0 [ recover ]
  ]
  ask balls [
    bounce
    collide
    shield
    event 
    ;; ball reports status by setting own label
    ifelse length ball-report > 0 [set label ball-report] [set label ""]
    ;; add any barriers to the route
    if label != "" [
      set route sentence route filter is-barrier? label
    ]
  ]
  ;; monitor status of the ball by checking value of its label
  ask people [
    monitor
  ]
  ;; move ball on
  ask balls [
    fd 1 
  ]
  repaint
  tick 
end

to-report aggregate-causals 
  let a-c n-values count barriers [0]
  let aggregate []
  let firsts []
  foreach sort people [set aggregate sentence aggregate [cbas] of ?] ;; with sentence, items are included in the result directly
  ;; not sure how this works (use of fput [] seems essential, and switching ?2 and ?1)
  set firsts reduce [fput (item 0 ?2) ?1] (fput [] aggregate)
  ;; occurences of each barrier
  let b-list sort-on [who] barriers
  let index 0
  foreach b-list [
    let occ occurrences ? firsts
    set a-c replace-item index a-c occ
    ask ? [set causals occ]    ;; update the barriers record of causals
    set index index + 1  
  ] 
  report a-c
end

;; monitor a ball in the vicinity and record objects (barriers and people) it comes into contact with 
to monitor
  ifelse any? balls in-radius mrad 
  [
    if show-labels [set label "!!"]     ;; display exclamation mark
    ifelse record = []
    [
      ;; start a new record
      let ballfollowed one-of balls in-radius mrad
      set record (list ballfollowed)
      ;; use sentence in order to include elements directly
      if [label] of ballfollowed != "" [set record sentence record [label] of ballfollowed]
    ]
    [
       ;; check ball id, check label and add new item to record 
       let ballfollowed item 0 record
       if [label] of ballfollowed != "" 
       [
         set record sentence record [label] of ballfollowed
       ]
    ]
  ] 
  [
    ;; when there are no balls in sight check record and reset it
    if record != [] [
      query-record
      set record []
      set label "" 
    ]
  ] 
end

;; detect if any barrier was struck followed by a person
to query-record
  ;; ignore dead barriers - when a turtle dies it turns into nobody -  filter these out
  let xcc map [? = nobody] record
  if (member? true xcc) = true [set record filter [? != nobody] record] 
  ;; if last item is barrier, ignore it. sublist args: first position (inclusive) and the second position (exclusive)
  if is-barrier? last record [set record sublist record 0 (length record - 1)]   
  ;; create logical map of barrier and find list item numbers
  let bmap map [is-barrier? ?] record    ;; map of barriers
  let bps positions-where-equal true bmap
  let recbar filter [[breed] of ? = barriers] record
  if length recbar > 0 [ 
    let lastp 0   ;; ball is included - first item of first seg
    foreach bps [
      let seg sublist record lastp ?
      set lastp ?
      ;; if the first item is a barrier then add causal
      if is-barrier? item 0 seg [
        foreach sublist seg 1 length seg [
          ;; print (word "causal link between " item 0 seg " and " ?)
          set cbas sentence cbas (list list item 0 seg ?)
        ]
      ]
    ]
    ;;last seg same code as above
    let fseg sublist record lastp length record
    ;; if the first item is a barrier then add causal
    if is-barrier? item 0 fseg [
        foreach sublist fseg 1 length fseg [
          ;; print (word "causal link between " item 0 fseg " and " ?)
          set cbas sentence cbas (list list item 0 fseg ?)
        ]
    ]
  ]
end

;; how to retrieve one-of position in a list
to-report positions-where-equal [x xs]
  let indices n-values length xs [?]
  report filter [x = item ? xs] indices
end 

;; count the number of occurrences of an item in a list
to-report occurrences [x the-list]
  report reduce
    [ifelse-value (?2 = x) [?1 + 1] [?1]] (fput 0 the-list)
end

to recover
  if damage > 0 and ticks - latest > recovery [ set damage damage - 1 ]
end

;; mitigate is done only when damage is zero. 
to mitigate
  let print-on true
  ;; relocate - move to a less dangerous place - if own history shows more damage on that patch than that of neighbours - and if one available
  if relocation-rule != "Off" and random-float 1 < random-relocate [
    if total-here > 2 + (max [total-here] of link-neighbors) [ 
    ;; if total-here > 1.2 * (max [total-here] of link-neighbors) [ 
      let ev-plots 0
      ;; move a neighboring patch, i.e. move a distance one unit 
      if relocation-rule = "patch-neighbour" [set ev-plots patch-set filter [member? ? habitable] sort neighbors with [not any? turtles-here]] 
      ;; move to patch next to one of link-neighbours
      if relocation-rule = "link-neighbour" [set ev-plots patch-set filter [member? ? habitable] sort (patch-set [neighbors] of link-neighbors) with [not any? turtles-here] ]
      ;; move to random patch
      if relocation-rule = "random" [set ev-plots habitable with [not any? turtles-here]]
      if count ev-plots > 0
      [
        move-to one-of ev-plots   ;; move to one of available patches, leaving any barriers behind ..
        if print-on [print (word "Person-" who " moved, total-here: " total-here)]
        set total-here 0
        set relocs relocs + 1
      ] 
    ]  
  ]
  ;; build a barrier with a small probability
  if build-barriers = true and random-float 1 < random-build [
    let ba-plots patch-set sort neighbors with [not any? turtles-here]  ;; allow outside of habitable
    ;; let ba-plots patch-set filter [member? ? habitable] sort neighbors with [not any? turtles-here]
    if count ba-plots > 0
    [
      ;; careful with hatch (heading and label) - each new turtle inherits of all its variables from its parent. 
      hatch-barriers 1 [           ;; hatch because turtle context
        set color 34 ;
        move-to one-of ba-plots 
        set owner myself           ;; myself refers to a person
        if print-on [print (word "Person-" [who] of myself " built barrier")]
        set end-of-life ticks + barrier-lifespan
        set times 0
        set causals 0
        set label ""
        ;; check if neighbouring barrier, set same orientation
        ifelse any? barriers-on neighbors4
        [ set heading [heading] of one-of barriers-on neighbors4]
        [set heading random 360]
      ]
    ]
  ] 
end

to collide
  ;; collide with another ball
  if any? other balls-here [
    set heading random 360 
  ]
end

;; potentially damaging event can occur if the ball evades any shield  
;; TODO non-susceptibles may be susceptible to peturbations
to event
  ;; setting the radius to half the size of ball - makes a big difference in calculating contact with person or barrier 
  let bradius size / 2
  ask people in-radius bradius [
  ;; ask people in-radius size [
    if susceptibility = 1 [
      set damage damage + 1
      set latest ticks
      set total total + 1
      set total-here total-here + 1
      set number-of-collisions number-of-collisions + 1
    ]
    
  ]
end

;; barriers can protect an area behind it from possible impact - can deflect or divert the ball 
;; TODO peturbation might have a percentage chance of destroying a barrier
to shield
  
  ;; setting the radius to half the size of ball - makes a big difference in calculating contact with person or barrier 
  ;;let bradius size / 2 
  ;;if any? barriers in-radius bradius [
  if any? barriers in-radius size [
    ifelse size = 2 [  ask barriers with [end-of-life > 0] in-radius size [die]] ;; peturbation causes (non-initial) barriers destruction
    [
      let b min-one-of barriers in-radius size [distance myself]
      ;;let b min-one-of barriers in-radius bradius [distance myself]
      ask b [set times times + 1]
      if barrier-action = "deflect"
      [
        ;; ball takes the heading of the barrier - with random adjustment
        set heading [heading] of b - 3 + random 7      
      ]
      if barrier-action = "deflect deterministic"
      [
        ;; ball takes the precise heading of the barrier
        set heading [heading] of b    
      ]
      if barrier-action = "deflect random"
      [ set heading random 360 ] ;; ball takes totally random deflection
    ]
  ]
end  

to bounce  ;; balls procedure
  ;; bounce off left and right walls
  if abs pxcor = max-pxcor
    [ set heading (- heading) fd 1]
  ;; bounce off top and bottom walls
  if abs pycor = max-pycor
    [ set heading (180 - heading) fd 1 ]
end

to repaint
  ask people [
    set color 34 + damage
  ]
  ifelse show-links?
  [ask links [show-link]]
  [ask links [hide-link]]
end

;; report any person (or barrier) the ball is in contact with
to-report ball-report
  let val []
  set val sort people in-radius size with [susceptibility = 1]
  if any? barriers-here [set val lput one-of barriers-here val]
  report val
end
@#$#@#$#@
GRAPHICS-WINDOW
210
10
649
470
16
16
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
1
1
1
ticks
30.0

BUTTON
9
370
87
403
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
9
406
72
439
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
665
10
865
160
time series of damage
time
no. of events
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot number-of-collisions"

CHOOSER
7
14
203
59
barrier-action
barrier-action
"deflect" "deflect random" "deflect deterministic" "protect"
0

PLOT
666
320
866
470
histogram of damage
damage
no. of people
0.0
5.0
0.0
10.0
true
false
";;set-plot-x-range 0 1\\n;;set-plot-y-range 0 10\\nset-plot-pen-mode 1\\n;;histogram map [position ? [red green blue]] ([color] of turtles)\\n;;set-histogram-num-bars 5" ""
PENS
"default" 1.0 0 -16777216 true "" "histogram [damage] of people"

PLOT
664
167
978
312
cum.damage for each agent (ordered)
NIL
times
0.0
10.0
0.0
10.0
true
false
"" "clear-plot\\nset-plot-pen-mode 1\\nlet plist sort-on [total] people\\nlet xs 0\\nforeach plist [\\n  plotxy xs [total] of ?\\n  set xs xs + 1\\n  ]"
PENS
"default" 1.0 0 -16777216 true "" ""

SWITCH
7
464
166
497
build-barriers?
build-barriers?
1
1
-1000

TEXTBOX
9
444
78
463
Mitigate
12
0.0
1

BUTTON
9
333
120
366
NIL
setup-balls
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
874
11
1114
161
total times for each barrier
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" "clear-plot\\nset-plot-pen-mode 1\\nlet blist sort-on [who] barriers\\nlet xs 0\\nforeach blist [\\n  plotxy xs [times] of ?\\n  set xs xs + 1\\n  ]"
PENS
"pen-0" 1.0 0 -7500403 true "" ""

PLOT
1119
10
1358
160
total causals for each barrier
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" "clear-plot\\nset-plot-pen-mode 1\\nlet blist sort-on [who] barriers\\nlet occlist aggregate-causals\\nlet xs 0\\nforeach blist [\\n  plotxy xs item xs occlist\\n  set xs xs + 1\\n  ]"
PENS
"default" 1.0 0 -7500403 true "" ""

SLIDER
8
540
181
573
tolerance
tolerance
0
1
0.1
0.01
1
NIL
HORIZONTAL

SLIDER
8
501
181
534
random-build
random-build
0
0.1
0.05
0.01
1
NIL
HORIZONTAL

CHOOSER
7
178
100
223
chosen-mrad
chosen-mrad
0 1 2 3 4
2

TEXTBOX
108
182
245
210
monitoring\\nability (0=none)
10
0.0
1

CHOOSER
8
281
186
326
chosen-max-recovery
chosen-max-recovery
60 80 100 500
1

CHOOSER
8
233
182
278
chosen-min-recovery
chosen-min-recovery
10 20 100
1

SWITCH
520
480
644
513
show-labels
show-labels
0
1
-1000

SLIDER
8
577
180
610
lifespan?
lifespan?
0
20000
10000
1000
1
NIL
HORIZONTAL

CHOOSER
208
527
346
572
network-type?
network-type?
"local" "random"
1

SWITCH
507
517
645
550
show-links?
show-links?
1
1
-1000

SLIDER
208
576
385
609
random-relocate
random-relocate
0
0.5
0.25
0.05
1
NIL
HORIZONTAL

INPUTBOX
6
66
167
126
number-of-people?
50
1
0
Number

CHOOSER
6
130
144
175
habitable?
habitable?
"single" "double" "twin" "four"
0

CHOOSER
209
476
378
521
relocation-rule?
relocation-rule?
"Off" "random" "link-neighbour" "patch-neighbour"
0

BUTTON
80
407
200
440
NIL
perturbate
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
873
321
1114
471
time series % ppl damaged
time
%
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if number-of-people > 0 [plot 100 * count people with [damage > 0] / number-of-people]"

PLOT
985
168
1248
314
cum.relocation for each agent
NIL
times
0.0
10.0
0.0
10.0
true
false
"" "clear-plot\\nset-plot-pen-mode 1\\nlet plist sort-on [relocs] people\\nlet xs 0\\nforeach plist [\\n  plotxy xs [relocs] of ?\\n  set xs xs + 1\\n  ]"
PENS
"default" 1.0 0 -16777216 true "" ""

@#$#@#$#@
## WHAT IS IT?

A bouncing, ricocheting ball represents a hazard that can cause damage to individuals it comes into contact with. We simulate individuals recovery and mitigation processes (including evasion and creation of barriers) and investigate the factors which affect the damage received and its distribution among individuals. The model could be used to investigate cooperative strategies in relation to recovery and mitigation, which depend upon using social networks (sharing resources, sharing information) and how the networks are altered when members experience great hazards and damage, or conversely, safety.

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

barrier
true
0
Line -7500403 true 150 0 150 300
Polygon -7500403 true true 150 0 90 90 150 60 210 90 150 0

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.0.3
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 1.0 0.0
0.0 1 1.0 0.0
0.2 0 1.0 0.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180

@#$#@#$#@
0
@#$#@#$#@
