;name agents
breed [young_ones young_one]
breed [young_adults young_adult]
breed [middle_aged_adults middle_aged_adult]
breed [seniors senior]

; variables that can be updated/changed as agents experience changes to themselves or the environment
globals [
  num-healthy
  num-infected
  num-recovered
  num-dead
  num-healthy-young-ones
  num-healthy-young-adults
  num-healthy-middle-aged-adults
  num-healthy-seniors
  num-infected-young-ones
  num-infected-young-adults
  num-infected-middle-aged-adults
  num-infected-seniors
  num-recovered-young-ones
  num-recovered-young-adults
  num-recovered-middle-aged-adults
  num-recovered-seniors
  num-dead-young-ones
  num-dead-young-adults
  num-dead-middle-aged-adults
  num-dead-seniors
  transmission-rate
  incubation-period
  symptomatic-period
  Log-data
  total-contact-rate
  contact-rate
]
; variable unique to each agents
turtles-own [
  healthy?
  infected?
  recovered?
  dead?
  susceptibility
  critical-illness-rate
  age-group
  days-infected
  days-since-symptoms
  daily-contacts
  vaccinated?
  wearing-mask?
  secondary-infections ; Note: variable to track secondary infections
]



to setup
  clear-all

  ;assign values to relevant global variables
  set transmission-rate 0.0089
  set incubation-period 122.4
  set symptomatic-period 240
  set Log-data []
  set total-contact-rate 0

  ; Note: let is used to create local variables that are used only within the current block of code.
  ; set number of agents as proportions of the total population
  let num-young-ones round (total-turtles * young-ones-proportion)
  let num-young-adults round (total-turtles * young-adults-proportion)
  let num-middle-aged-adults round (total-turtles * middle-aged-adults-proportion)
  let num-seniors round (total-turtles * seniors-proportion)

  ; create locations
  create-locations

  ; create agents, define name, set agent specific unique variables/properties
  create-turtles num-young-ones [
    set breed young_ones
    set age-group "young ones"
    set-turtle-properties young-ones-susceptibility young-ones-vaccination-rate young-ones-masking-rate young-ones-contact-rate
  ]
  create-turtles num-young-adults [
    set breed young_adults
    set age-group "young adults"
    set-turtle-properties young-adults-susceptibility young-adults-vaccination-rate young-adults-masking-rate young-adults-contact-rate
  ]
  create-turtles num-middle-aged-adults [
    set breed middle_aged_adults
    set age-group "middle aged adults"
    set-turtle-properties middle-aged-adults-susceptibility middle-aged-adults-vaccination-rate middle-aged-adults-masking-rate middle-aged-adults-contact-rate
  ]
  create-turtles num-seniors [
    set breed seniors
    set age-group "seniors"
    set-turtle-properties seniors-susceptibility seniors-vaccination-rate seniors-masking-rate seniors-contact-rate
  ]

  ;set number of intially infected
  repeat initial-infected [
    ask one-of turtles with [not infected?] [
      set infected? true
      set healthy? false
      set days-infected 0
      set color red
    ]
  ]

  reset-ticks
end

; assign the unique variables to agents (boolean, integer, float)
to set-turtle-properties [initial-susceptibility vaccination-rate masking-rate initial-contact-rate]
  setxy random-xcor random-ycor
  set healthy? true
  set infected? false
  set recovered? false
  set dead? false
  set days-infected 0
  set days-since-symptoms 0
  set daily-contacts 0
  set vaccinated? (random-float 1 < vaccination-rate)
  set wearing-mask? (random-float 1 < masking-rate)
  set susceptibility initial-susceptibility
  set contact-rate initial-contact-rate
  set secondary-infections 0

  if vaccinated? [
    set susceptibility susceptibility * (1 - vaccination-effectiveness) ; Adjust susceptibility based on vaccination effectiveness
  ]

  if age-group = "young ones" [
    set critical-illness-rate 0.0004
    set color yellow
  ]
  if age-group = "young adults" [
    set critical-illness-rate 0.0029
    set color blue
  ]
  if age-group = "middle aged adults" [
    set critical-illness-rate 0.0039
    set color green
  ]
  if age-group = "seniors" [
    set critical-illness-rate 0.2
    set color violet
  ]
end

to create-locations
  ask patches [
    if random-float 1 < 0.1 [set pcolor blue]
    if random-float 1 < 0.05 [set pcolor yellow]
    if random-float 1 < 0.03 [set pcolor orange]
    if random-float 1 < 0.05 [set pcolor green]
    if random-float 1 < 0.04 [set pcolor magenta]
    if random-float 1 < 0.02 [set pcolor black]
    if random-float 1 < 0.03 [set pcolor white]
  ]
end
; define movement pattern based on states and age group
to move-to-location
  if infected? [
    if days-infected >= incubation-period [
      ifelse days-since-symptoms > 0 and days-since-symptoms <= symptomatic-period [
        ifelse random 2 = 0 [
          move-to one-of patches with [pcolor = magenta]
        ] [
          move-to one-of patches with [pcolor = black]
        ]
      ] [
        move-to one-of patches with [pcolor = magenta]
      ]
    ]
  ]
  if not infected? [
    let current-hour ticks mod 24
    let current-day (ticks / 24) mod 7


      ifelse current-day < 5 [
        if age-group = "young ones" [
          if current-hour >= 8 and current-hour < 9 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 9 and current-hour < 15 [move-to one-of patches with [pcolor = orange]]
          if current-hour >= 15.5 and current-hour < 17 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 17 and current-hour < 18 [move-to one-of patches with [pcolor = orange]]
          if current-hour >= 18 and current-hour < 19 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 19 and current-hour < 20 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 20 and current-hour < 21 [move-to one-of patches with [pcolor = blue]]
        ]
        if age-group = "young adults" [
          if current-hour >= 7 and current-hour < 8 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 8 and current-hour < 9 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 9 and current-hour < 17 [move-to one-of patches with [pcolor = yellow]]
          if current-hour >= 17.5 and current-hour < 18.5 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 18.5 and current-hour < 19.5 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 19.5 and current-hour < 21 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 21 and current-hour < 22 [move-to one-of patches with [pcolor = blue]]
        ]
        if age-group = "middle aged adults" [
          if current-hour >= 6 and current-hour < 7 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 7 and current-hour < 8 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 8 and current-hour < 17 [move-to one-of patches with [pcolor = yellow]]
          if current-hour >= 17.5 and current-hour < 18.5 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 18.5 and current-hour < 19.5 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 19.5 and current-hour < 21 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 21 and current-hour < 22 [move-to one-of patches with [pcolor = blue]]
        ]
        if age-group = "seniors" [
          if current-hour >= 7 and current-hour < 8 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 8 and current-hour < 9 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 9 and current-hour < 12 [move-to one-of patches with [pcolor = yellow]]
          if current-hour >= 12 and current-hour < 13 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 13 and current-hour < 15 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 15 and current-hour < 17 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 17 and current-hour < 18 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 18 and current-hour < 19 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 19 and current-hour < 21 [move-to one-of patches with [pcolor = blue]]
        ]
        if age-group = "elderly" [
          if current-hour >= 8 and current-hour < 9 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 9 and current-hour < 11 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 11 and current-hour < 13 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 13 and current-hour < 14 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 14 and current-hour < 16 [move-to one-of patches with [pcolor = white]]
          if current-hour >= 16 and current-hour < 18 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 18 and current-hour < 19 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 19 and current-hour < 21 [move-to one-of patches with [pcolor = blue]]
        ]
      ] [
        if age-group = "young ones" [
          if current-hour >= 8 and current-hour < 9 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 9 and current-hour < 11 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 11 and current-hour < 12 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 12 and current-hour < 13 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 13 and current-hour < 15 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 15 and current-hour < 17 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 17 and current-hour < 18 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 18 and current-hour < 19 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 19 and current-hour < 20 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 20 and current-hour < 21 [move-to one-of patches with [pcolor = blue]]
        ]
        if age-group = "young adults" [
          if current-hour >= 8 and current-hour < 9 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 9 and current-hour < 11 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 11 and current-hour < 13 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 13 and current-hour < 15 [move-to one-of patches with [pcolor = white]]
          if current-hour >= 15 and current-hour < 17 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 17 and current-hour < 18 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 18 and current-hour < 20 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 20 and current-hour < 22 [move-to one-of patches with [pcolor = white]]
        ]
        if age-group = "middle aged adults" [
          if current-hour >= 7 and current-hour < 8 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 8 and current-hour < 9 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 9 and current-hour < 17 [move-to one-of patches with [pcolor = yellow]]
          if current-hour >= 17.5 and current-hour < 18.5 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 18.5 and current-hour < 19.5 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 19.5 and current-hour < 21 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 21 and current-hour < 22 [move-to one-of patches with [pcolor = blue]]
        ]
        if age-group = "seniors" [
          if current-hour >= 8 and current-hour < 9 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 9 and current-hour < 11 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 11 and current-hour < 13 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 13 and current-hour < 14 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 14 and current-hour < 16 [move-to one-of patches with [pcolor = white]]
          if current-hour >= 16 and current-hour < 18 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 18 and current-hour < 19 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 19 and current-hour < 21 [move-to one-of patches with [pcolor = blue]]
        ]
        if age-group = "elderly" [
          if current-hour >= 8 and current-hour < 9 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 9 and current-hour < 11 [move-to one-of patches with [pcolor = green]]
          if current-hour >= 11 and current-hour < 13 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 13 and current-hour < 14 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 14 and current-hour < 16 [move-to one-of patches with [pcolor = white]]
          if current-hour >= 16 and current-hour < 18 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 18 and current-hour < 19 [move-to one-of patches with [pcolor = blue]]
          if current-hour >= 19 and current-hour < 21 [move-to one-of patches with [pcolor = blue]]
        ]
      ]
    ]
    setxy random-xcor random-ycor

end

; define general movement pattern and behaviours
to go
  ask turtles [
    if not dead? [
      move-to-location
      infect-others
      develop-symptoms
      recover-or-die
    ]
  ]
  update-counts
  if num-infected = 0 [
    calculate-R0
    stop
  ]
  log-contact-rates
  tick
end

to calculate-R0
  let total-secondary-infections sum [secondary-infections] of turtles with [infected? or recovered? or dead?]
  let total-infected count turtles with [infected? or recovered? or dead?]
  if total-infected > 0 [
    let R0 total-secondary-infections / total-infected
    print (word "R0: " R0)
  ]
end

; define how infected agents spread the disease
to infect-others
  if infected? [
    ;define infectious period
    let infectious-start incubation-period - 24
    let infectious-end incubation-period + symptomatic-period + 72

    ;check if within infected period and define infection radius
    if days-infected >= infectious-start and days-infected <= infectious-end [
      let infection-radius 1
     ;if practicing-social-distancing? [
      ;  set infection-radius 0.2
     ; ]

      ;define set number and "condition" of likely to be infected
      let nearby-turtles turtles in-radius infection-radius with [not dead? and not infected?]
      ask n-of (count nearby-turtles * contact-rate) nearby-turtles [
        let adjusted-transmission-rate transmission-rate
        if wearing-mask? [
          set adjusted-transmission-rate transmission-rate * (1 - masking-effectiveness)
        ]
        ; if radom-float < adjusted-transmission-rate * susceptibility, get infected.
        if random-float 1 < adjusted-transmission-rate * susceptibility [
          set infected? true
          set healthy? false
          set days-infected 0
          set color red
          ask myself [set secondary-infections secondary-infections + 1]
        ]

        ; keep track of conatacts
        set daily-contacts daily-contacts + 1
        ask myself [set daily-contacts daily-contacts + 1]
      ]
    ]
  ]
end

; define when infections should start, it will indicate when infected agents should isolate themselves
to develop-symptoms
  if infected? [
    if days-infected = incubation-period [
      set days-since-symptoms 0
    ]
    set days-since-symptoms days-since-symptoms + 1
  ]
end

; define conditions for recovery or death
to recover-or-die
  if infected? [
    set days-infected days-infected + 1

    if days-infected > incubation-period [
      set days-since-symptoms days-since-symptoms + 1
    ]

    if days-since-symptoms >= symptomatic-period [
      if random-float 1 < critical-illness-rate [
        ifelse random-float 1 < 0.5 [
          set dead? true
          set infected? false
          set healthy? false
          set recovered? false
          set color black
        ] [
          set recovered? true
          set infected? false
          set healthy? false
          set dead? false
          set color gray
        ]
      ]

      ;fail safe to ensure that all agents that didnt die, did recover.
      if not dead? [
        set recovered? true
        set infected? false
        set healthy? false
        set dead? false
        set color gray
      ]
    ]

    ;recover after the symptomatic period has passed
    if days-infected > incubation-period + symptomatic-period [
      set recovered? true
      set infected? false
      set healthy? false
      set dead? false
      set color gray
    ]
  ]
end

;update counts
to update-counts
  set num-healthy count turtles with [healthy?]
  set num-infected count turtles with [infected?]
  set num-recovered count turtles with [recovered?]
  set num-dead count turtles with [dead?]

  set num-healthy-young-ones count young_ones with [healthy?]
  set num-healthy-young-adults count young_adults with [healthy?]
  set num-healthy-middle-aged-adults count middle_aged_adults with [healthy?]
  set num-healthy-seniors count seniors with [healthy?]

  set num-infected-young-ones count young_ones with [infected?]
  set num-infected-young-adults count young_adults with [infected?]
  set num-infected-middle-aged-adults count middle_aged_adults with [infected?]
  set num-infected-seniors count seniors with [infected?]

  set num-recovered-young-ones count young_ones with [recovered?]
  set num-recovered-young-adults count young_adults with [recovered?]
  set num-recovered-middle-aged-adults count middle_aged_adults with [recovered?]
  set num-recovered-seniors count seniors with [recovered?]

  set num-dead-young-ones count young_ones with [dead?]
  set num-dead-young-adults count young_adults with [dead?]
  set num-dead-middle-aged-adults count middle_aged_adults with [dead?]
  set num-dead-seniors count seniors with [dead?]

  ;optional, can be replaced with output from behaviour space
  set Log-data lput (list ticks num-healthy num-infected num-recovered num-dead
              num-healthy-young-ones num-healthy-young-adults num-healthy-middle-aged-adults num-healthy-seniors
              num-infected-young-ones num-infected-young-adults num-infected-middle-aged-adults num-infected-seniors
              num-recovered-young-ones num-recovered-young-adults num-recovered-middle-aged-adults num-recovered-seniors
              num-dead-young-ones num-dead-young-adults num-dead-middle-aged-adults num-dead-seniors) Log-data
end

to log-contact-rates
  let total-contacts sum [daily-contacts] of turtles
  set total-contact-rate total-contacts
  ask turtles [set daily-contacts 0]
end
@#$#@#$#@
GRAPHICS-WINDOW
122
10
559
448
-1
-1
13.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

BUTTON
25
24
88
57
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
28
64
91
97
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
565
17
743
50
young-ones-proportion
young-ones-proportion
0
1
0.2
0.05
1
NIL
HORIZONTAL

SLIDER
565
60
750
93
young-adults-proportion
young-adults-proportion
0
1
0.21
0.05
1
NIL
HORIZONTAL

SLIDER
566
103
781
136
middle-aged-adults-proportion
middle-aged-adults-proportion
0
1
0.37
0.05
1
NIL
HORIZONTAL

SLIDER
567
149
739
182
seniors-proportion
seniors-proportion
0
1
0.29
0.05
1
NIL
HORIZONTAL

MONITOR
35
266
104
311
NIL
num-dead
17
1
11

MONITOR
16
214
113
259
NIL
num-recovered
17
1
11

MONITOR
23
164
109
209
NIL
num-infected
17
1
11

MONITOR
23
112
105
157
NIL
num-healthy
17
1
11

SLIDER
759
16
949
49
young-ones-susceptibility
young-ones-susceptibility
0
1
0.38
0.01
1
NIL
HORIZONTAL

SLIDER
762
56
959
89
young-adults-susceptibility
young-adults-susceptibility
0
1
0.78
0.01
1
NIL
HORIZONTAL

SLIDER
792
101
1021
134
middle-aged-adults-susceptibility
middle-aged-adults-susceptibility
0
1
0.78
0.01
1
NIL
HORIZONTAL

SLIDER
796
152
968
185
seniors-susceptibility
seniors-susceptibility
0
1
0.79
0.01
1
NIL
HORIZONTAL

SLIDER
836
259
1008
292
social-distancing-rate
social-distancing-rate
0
1
0.0
0.1
1
NIL
HORIZONTAL

SLIDER
1025
14
1235
47
young-ones-vaccination-rate
young-ones-vaccination-rate
0
1
0.1
0.1
1
NIL
HORIZONTAL

SLIDER
1026
55
1243
88
young-adults-vaccination-rate
young-adults-vaccination-rate
0
1
0.0
0.01
1
NIL
HORIZONTAL

SLIDER
1029
99
1278
132
middle-aged-adults-vaccination-rate
middle-aged-adults-vaccination-rate
0
1
0.0
0.01
1
NIL
HORIZONTAL

SLIDER
1031
148
1215
181
seniors-vaccination-rate
seniors-vaccination-rate
0
1
0.0
0.01
1
NIL
HORIZONTAL

SLIDER
1300
14
1492
47
young-ones-masking-rate
young-ones-masking-rate
0
1
0.7
0.1
1
NIL
HORIZONTAL

SLIDER
1303
55
1501
88
young-adults-masking-rate
young-adults-masking-rate
0
1
0.86
0.1
1
NIL
HORIZONTAL

SLIDER
1287
99
1518
132
middle-aged-adults-masking-rate
middle-aged-adults-masking-rate
0
1
0.86
0.1
1
NIL
HORIZONTAL

SLIDER
1314
142
1486
175
seniors-masking-rate
seniors-masking-rate
0
1
0.98
0.1
1
NIL
HORIZONTAL

SLIDER
1276
254
1449
287
masking-effectiveness
masking-effectiveness
0
1
0.6
0.01
1
NIL
HORIZONTAL

SLIDER
1050
258
1244
291
vaccination-effectiveness
vaccination-effectiveness
0
1
0.0
0.01
1
NIL
HORIZONTAL

SLIDER
840
335
1012
368
total-turtles
total-turtles
0
2000
2000.0
100
1
NIL
HORIZONTAL

SLIDER
568
216
757
249
young-ones-contact-rate
young-ones-contact-rate
0
1
1.0
0.05
1
NIL
HORIZONTAL

SLIDER
570
265
766
298
young-adults-contact-rate
young-adults-contact-rate
0
1
1.0
0.1
1
NIL
HORIZONTAL

SLIDER
572
317
800
350
middle-aged-adults-contact-rate
middle-aged-adults-contact-rate
0
1
1.0
0.05
1
NIL
HORIZONTAL

SLIDER
574
369
746
402
seniors-contact-rate
seniors-contact-rate
0
1
1.0
0.05
1
NIL
HORIZONTAL

SWITCH
1282
333
1446
366
social-distancing-on
social-distancing-on
1
1
-1000

SLIDER
1070
332
1242
365
initial-infected
initial-infected
0
20
10.0
1
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This model was designed to study the spread of an infectious disease among different age groups. The model aims to track key health outcomes (healthy, infected, recovered and dead individuals) with distinct population groups (young ones, young adults, middle-aged adults, and seniors)

It also simulates interactions based on agent characteristics, contact rates and protective behaviours like vaccination and mask wearing.

## HOW IT WORKS

The model operates by using agent-based rules.Each agent is assigned an age group, health status, susceptibility to infection and other variables such as vaccination status or mask wearing. 

Agents move between pre-defined locations based on daily schedules and interact with other agents, leading to potential infections.

Agents transition between healthy, infected, symptomatic, recovered, or dead status based on their individual conditions and disease progression (e.g. incubation period, critical illness rate). 

The transmission of the disease depends on factors such as contact rates amd the use of protective measures (masking).

## HOW TO USE IT
To use the model, follow the following steps:

### Set-up: 

Click the "Set-up" button on the left to intialize the simulation. This prepares the population based on the settings you have configured using the sliders on the right.

### Start the simulation

Click "go" to start the simulation. The model will begin running, and the number of healthy, infected, recovered, and dead agents will update in the boxes to the left.

### Adjust the paramters

Use the sliders on the right to adjust the following parameters that influence the behaviour of the agents and disease spread:

#### Population proportions 
Adjust the sliders for the proportion of each age group
Susceptibility: Change how susceptible each age group is to the disease.

#### Contact rate
Modify the proportion of individuals that come in contact with each infected individual. A lower contact rate implies that less people come in contact with infected persons.

#### Masking and Vaccination rates
Control the percentage of each age group that wears masks and is vaccinated respectively.

#### Vaccination and Masking Effectiveness
Adjust the effectiveness of vaccines and masks in reducing susceptibility and transmission respectively.

#### Initial Infected
Set the number of initially infected agents.

#### Total-turtles
Set the total number of agents.

## THINGS TO NOTICE

As simulation runs, observe:

-The impact of varying contact rates, vaccination rates and mask usage on the spread of infection.

-How different age groups are affected based on their susceptibility and the effectiveness of interventions like masking and vaccination.

## THINGS TO TRY

### Experiment intervention strategies
Try setting different vaccination rates and masking effectiveness for various age groups and see how it impacts the spread.

### Simulate Different Social Scenarios
Increase or decrease the contact rates for specific age groups to model situations such as: school reopening (young ones) or return to work for (adults).

### Explore Outbreaks Scenarios
Adjust the initial the initial number of infected individuals and experiment with different population proportions to simulate outbreaks conditions in different types of communities.

## EXTENDING THE MODEL

To make the model more detailed, users could add more complex interactions, such as varying contact rates based on individual behaviours or introducing additional interventions (social distancing, hosptitals where patients come in contact with healthy health workers, elderly homes etc).

Further developments could aslo include environmental factors that affect transmission such as wind circulation, or integrating testing and isolation protocols.

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

The model utilizes several advanced NetLogo features, including agent-based movement, conditional decision making, and dynamic changes to agent states. 

The "turtles-own" functions tracks variables specific to each agents, while "ask" commands dictate agent behaviour. The model employs randomness for assigning infection status and protective measures, which could be controlled via sliders.

## RELATED MODELS

Similar models in the NetLogo library include epidemic spread that focus on the movement and interactions of agents. These models might help users explore different strategies to mitigate the spread of infectious diseases by comparing different types of interventions and population structures.

## CREDITS AND REFERENCES
You could find similar models at: modelingcommons.org
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.2
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="Baseline" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <exitCondition>count turtles with [infected?] = 0</exitCondition>
    <metric>count young_ones with [infected?]</metric>
    <metric>count young_adults with [infected?]</metric>
    <metric>count middle_aged_adults with [infected?]</metric>
    <metric>count seniors with [infected?]</metric>
    <metric>count young_ones with [recovered?]</metric>
    <metric>count young_adults with [recovered?]</metric>
    <metric>count middle_aged_adults with [recovered?]</metric>
    <metric>count seniors with [recovered?]</metric>
    <metric>count young_ones with [dead?]</metric>
    <metric>count young_adults with [dead?]</metric>
    <metric>count middle_aged_adults with [dead?]</metric>
    <metric>count seniors with [dead?]</metric>
    <metric>total-contact-rate</metric>
    <enumeratedValueSet variable="initial-infected">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-susceptibility">
      <value value="0.79"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-proportion">
      <value value="0.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-on">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-proportion">
      <value value="0.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-proportion">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masking-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-susceptibility">
      <value value="0.39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-proportion">
      <value value="0.13"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-turtles">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Masking" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <exitCondition>count turtles with [infected?] = 0</exitCondition>
    <metric>count young_ones with [infected?]</metric>
    <metric>count young_adults with [infected?]</metric>
    <metric>count middle_aged_adults with [infected?]</metric>
    <metric>count seniors with [infected?]</metric>
    <metric>count young_ones with [recovered?]</metric>
    <metric>count young_adults with [recovered?]</metric>
    <metric>count middle_aged_adults with [recovered?]</metric>
    <metric>count seniors with [recovered?]</metric>
    <metric>count young_ones with [dead?]</metric>
    <metric>count young_adults with [dead?]</metric>
    <metric>count middle_aged_adults with [dead?]</metric>
    <metric>count seniors with [dead?]</metric>
    <metric>total-contact-rate</metric>
    <enumeratedValueSet variable="initial-infected">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-susceptibility">
      <value value="0.79"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-proportion">
      <value value="0.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-on">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-proportion">
      <value value="0.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-proportion">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masking-effectiveness">
      <value value="0.56"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-susceptibility">
      <value value="0.39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-proportion">
      <value value="0.13"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-masking-rate">
      <value value="0.8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-turtles">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="vaccination" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <exitCondition>count turtles with [infected?] = 0</exitCondition>
    <metric>count young_ones with [infected?]</metric>
    <metric>count young_adults with [infected?]</metric>
    <metric>count middle_aged_adults with [infected?]</metric>
    <metric>count seniors with [infected?]</metric>
    <metric>count young_ones with [recovered?]</metric>
    <metric>count young_adults with [recovered?]</metric>
    <metric>count middle_aged_adults with [recovered?]</metric>
    <metric>count seniors with [recovered?]</metric>
    <metric>count young_ones with [dead?]</metric>
    <metric>count young_adults with [dead?]</metric>
    <metric>count middle_aged_adults with [dead?]</metric>
    <metric>count seniors with [dead?]</metric>
    <metric>total-contact-rate</metric>
    <enumeratedValueSet variable="initial-infected">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-susceptibility">
      <value value="0.79"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-proportion">
      <value value="0.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-vaccination-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-on">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-proportion">
      <value value="0.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-proportion">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-vaccination-rate">
      <value value="0.93"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-vaccination-rate">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination-effectiveness">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masking-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-vaccination-rate">
      <value value="0.86"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-susceptibility">
      <value value="0.39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-proportion">
      <value value="0.13"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-turtles">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Masking2" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <exitCondition>count turtles with [infected?] = 0</exitCondition>
    <metric>count young_ones with [infected?]</metric>
    <metric>count young_adults with [infected?]</metric>
    <metric>count middle_aged_adults with [infected?]</metric>
    <metric>count seniors with [infected?]</metric>
    <metric>count young_ones with [recovered?]</metric>
    <metric>count young_adults with [recovered?]</metric>
    <metric>count middle_aged_adults with [recovered?]</metric>
    <metric>count seniors with [recovered?]</metric>
    <metric>count young_ones with [dead?]</metric>
    <metric>count young_adults with [dead?]</metric>
    <metric>count middle_aged_adults with [dead?]</metric>
    <metric>count seniors with [dead?]</metric>
    <metric>total-contact-rate</metric>
    <enumeratedValueSet variable="initial-infected">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-susceptibility">
      <value value="0.66"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-susceptibility">
      <value value="0.66"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-proportion">
      <value value="0.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-on">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-proportion">
      <value value="0.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-proportion">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masking-effectiveness">
      <value value="0.49"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-susceptibility">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-susceptibility">
      <value value="0.66"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-proportion">
      <value value="0.13"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-masking-rate">
      <value value="0.8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-turtles">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Baseline2" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <exitCondition>count turtles with [infected?] = 0</exitCondition>
    <metric>count young_ones with [infected?]</metric>
    <metric>count young_adults with [infected?]</metric>
    <metric>count middle_aged_adults with [infected?]</metric>
    <metric>count seniors with [infected?]</metric>
    <metric>count young_ones with [recovered?]</metric>
    <metric>count young_adults with [recovered?]</metric>
    <metric>count middle_aged_adults with [recovered?]</metric>
    <metric>count seniors with [recovered?]</metric>
    <metric>count young_ones with [dead?]</metric>
    <metric>count young_adults with [dead?]</metric>
    <metric>count middle_aged_adults with [dead?]</metric>
    <metric>count seniors with [dead?]</metric>
    <metric>total-contact-rate</metric>
    <enumeratedValueSet variable="initial-infected">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-susceptibility">
      <value value="0.66"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-susceptibility">
      <value value="0.66"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-proportion">
      <value value="0.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-on">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-proportion">
      <value value="0.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-proportion">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masking-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-susceptibility">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-susceptibility">
      <value value="0.66"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-proportion">
      <value value="0.13"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-turtles">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="contact" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <exitCondition>count turtles with [infected?] = 0</exitCondition>
    <metric>count young_ones with [infected?]</metric>
    <metric>count young_adults with [infected?]</metric>
    <metric>count middle_aged_adults with [infected?]</metric>
    <metric>count seniors with [infected?]</metric>
    <metric>count young_ones with [recovered?]</metric>
    <metric>count young_adults with [recovered?]</metric>
    <metric>count middle_aged_adults with [recovered?]</metric>
    <metric>count seniors with [recovered?]</metric>
    <metric>count young_ones with [dead?]</metric>
    <metric>count young_adults with [dead?]</metric>
    <metric>count middle_aged_adults with [dead?]</metric>
    <metric>count seniors with [dead?]</metric>
    <metric>total-contact-rate</metric>
    <enumeratedValueSet variable="middle-aged-adults-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-susceptibility">
      <value value="0.79"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-proportion">
      <value value="0.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-on">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-proportion">
      <value value="0.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-proportion">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masking-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-contact-rate">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-susceptibility">
      <value value="0.39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-proportion">
      <value value="0.13"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-turtles">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-contact-rate">
      <value value="0.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <metric>count turtles</metric>
    <enumeratedValueSet variable="middle-aged-adults-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-susceptibility">
      <value value="0.79"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="initial-infected">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-proportion">
      <value value="0.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-on">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-proportion">
      <value value="0.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-proportion">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="social-distancing-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masking-effectiveness">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-vaccination-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-contact-rate">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-susceptibility">
      <value value="0.39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-proportion">
      <value value="0.13"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-masking-rate">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-turtles">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-contact-rate">
      <value value="0.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Combined" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <exitCondition>count turtles with [infected?] = 0</exitCondition>
    <metric>count young_ones with [infected?]</metric>
    <metric>count young_adults with [infected?]</metric>
    <metric>count middle_aged_adults with [infected?]</metric>
    <metric>count seniors with [infected?]</metric>
    <metric>count young_ones with [recovered?]</metric>
    <metric>count young_adults with [recovered?]</metric>
    <metric>count middle_aged_adults with [recovered?]</metric>
    <metric>count seniors with [recovered?]</metric>
    <metric>count young_ones with [dead?]</metric>
    <metric>count young_adults with [dead?]</metric>
    <metric>count middle_aged_adults with [dead?]</metric>
    <metric>count seniors with [dead?]</metric>
    <metric>total-contact-rate</metric>
    <enumeratedValueSet variable="initial-infected">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-susceptibility">
      <value value="0.79"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-proportion">
      <value value="0.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-vaccination-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-proportion">
      <value value="0.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-proportion">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-vaccination-rate">
      <value value="0.93"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-vaccination-rate">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination-effectiveness">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masking-effectiveness">
      <value value="0.56"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-vaccination-rate">
      <value value="0.86"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-susceptibility">
      <value value="0.39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-contact-rate">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-proportion">
      <value value="0.13"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-masking-rate">
      <value value="0.8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-turtles">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-contact-rate">
      <value value="0.5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Combined2" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <timeLimit steps="3000"/>
    <exitCondition>count turtles with [infected?] = 0</exitCondition>
    <metric>count young_ones with [infected?]</metric>
    <metric>count young_adults with [infected?]</metric>
    <metric>count middle_aged_adults with [infected?]</metric>
    <metric>count seniors with [infected?]</metric>
    <metric>count young_ones with [recovered?]</metric>
    <metric>count young_adults with [recovered?]</metric>
    <metric>count middle_aged_adults with [recovered?]</metric>
    <metric>count seniors with [recovered?]</metric>
    <metric>count young_ones with [dead?]</metric>
    <metric>count young_adults with [dead?]</metric>
    <metric>count middle_aged_adults with [dead?]</metric>
    <metric>count seniors with [dead?]</metric>
    <metric>total-contact-rate</metric>
    <enumeratedValueSet variable="initial-infected">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-susceptibility">
      <value value="0.79"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-proportion">
      <value value="0.37"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-vaccination-rate">
      <value value="0.7"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-proportion">
      <value value="0.29"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-proportion">
      <value value="0.21"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-vaccination-rate">
      <value value="0.93"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-vaccination-rate">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="vaccination-effectiveness">
      <value value="0.98"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="masking-effectiveness">
      <value value="0.49"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-contact-rate">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-masking-rate">
      <value value="0.6"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-adults-vaccination-rate">
      <value value="0.86"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-susceptibility">
      <value value="0.39"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-contact-rate">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="middle-aged-adults-susceptibility">
      <value value="0.78"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="young-ones-proportion">
      <value value="0.13"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-masking-rate">
      <value value="0.8"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="total-turtles">
      <value value="2000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="seniors-contact-rate">
      <value value="0.5"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
