extensions [ matrix r ]

globals [
  Free-Interest-Rate
  GDP
  Unemployment                                                ; sum of the households without a job
  Consumption                                                 ; sum of goods sold by the consumptions firms
  Total-Investment                                            ; sum of total capital sold by the capital production firms
  Goods-Price-Level                                           ; sum (price * qt) / Qt sold
  Goods-Price-Level_t-1
  Capital-Price-Level                                         ; sum (price * qt) / Qt sold
  Capital-Price-Level_t-1
  Total-Loans                                                 ; sum of loans
  Total-Loans-C-Firms                                         ; sum of loans with the C-Firms
  Total-Loans-K-Firms                                         ; sum of loans with the K-Firms
  C-Firms                                                     ; group of firms with property consumption production
  K-Firms                                                     ; group of firms with property capital production
  Workers                                                     ; group of households with property worker
  Capitalists                                                 ; group of households with property capitalist
  Retail-Banks                                                ; group of banks with property retail
  Wholesale-Banks                                             ; group of banks with property wholeseler
  C-Broke                                                     ; sum of Consumption firms that went broke last period
  K-Broke                                                     ; sum of capital production firms that went broke last period
  Matrix-C-Firms-Bankruptcy                                   ; input with information that will be used in calculation into R
  Matrix-K-Firms-Bankruptcy                                   ;input with information that will be used in calculation into R
  Beta_1-C-Firms                                              ; parameter utilized by the banks to calculate the interest rate for the firms
  Beta_2-C-Firms                                              ; parameter utilized by the banks to calculate the interest rate for the firms
  Beta_1-K-Firms                                              ; parameter utilized by the banks to calculate the interest rate for the firms
  Beta_2-K-Firms                                              ; parameter utilized by the banks to calculate the interest rate for the firms
  Mi
  Mj
  Mjk
  ]

breed               [Households Household]                    ; agent class
breed               [Firms      Firm     ]                    ; agent class
breed               [Banks      Bank     ]                    ; agent class
directed-link-breed [Hireds     Hired    ]                    ; link used to link firms to the worker
directed-link-breed [Owners     Owner    ]                    ; link used to link firms to their owner/capitalist

Households-own [
  Worker?                                                     ; property of the household, yes if it is a worker, not if it is a capitalis
  Human-Wealth                                                ; parameter to calibrate how much of the intire wealth will be use for consumption
  Current-Income                                              ; the salary if it is a worker, the profit if it is a capitalist
  Deposit-Bank                                                ; the total of wealth of the households held with the bank
  Desire-Consume                                              ; theorical value the household want to consume this period
  Consume                                                     ; real value the household did consume
  Price-List                                                  ; list with the prices of firms the household visited last period
  ]

Firms-own [
  C-Firms?                                                    ; property of the firms, yes if it is a consumption production firm, not if is a capital production firm

  Price                                                       ; last price praticed by the firm
  Demand                                                      ; last quantity demanded
  Production                                                  ; actual production of the firm
  Stock                                                       ; amount availible for sales
  Delta                                                       ; parameter used to calculate new price
  Number-Employee                                             ; actual number of employee of this firm
  Desire-Employee                                             ; ideal number of employee for this firm
  Vacancy                                                     ; number of opened places at this firm for new workers

  Revenue                                                     ; sum of quantity sold times the price
  Salles                                                      ; sum of quantity sold last period
  Profit                                                      ; revenue less the despenses
  Dividend                                                    ; part of the profit given to the owner
  Need-Cash                                                   ; money that will be ask at bank as a new loan
  Bank-Deposit                                                ; money availible at the bank
  Leverage                                                    ; assets of firm divided by their debts
  Broke                                                       ; true or false indicator, yes if the firm went broke this period
  Assets                                                      ; sum deposit at bank and stock availible for sales
  Current-Retail-Loan                                         ; sum of loans with the retail bank
  Interest-Retail                                             ; actual interest rate praticed with this firm by the retail bank
  Current-Wholesale-Loan                                      ; sum of loans with the wholesaler bank
  Interest-Wholesale                                          ; actual interest rate praticed with this firm by the wholesale bank
  Retail-Bankruptcy                                           ; the probability to go bankruptcy by this firm, calculated by the retail bank
  Wholesale-Bankruptcy                                        ; the probability to go bankruptcy by this firm, calculated by the wholesaler bank
  Interest-Pay                                                ; sum of interest rate payed last period
  Installment-Pay                                             ; sum of the installment payed last period
  T-expected                                                  ; the expected life expan for this firm

  Capital                                                     ; property of the C-firms, indicate the capital availible to be used at the production process
  Desire-Investment                                           ; property of the C-Firms, ideal investiment needed
  Price-List                                                  ; property of the C-Firms, list of prices from the K-Firms visited last period trying to buy capital goods
  Investment                                                  ; property of the C-Firms, capital goods bough last period
  Investment-Memory                                           ; property of the C-firms, parameter utilized to calculate te Desire-Investment
  Util-Capacity                                               ; property of the C-Firms, indicates how much of the Capital had been used

  Stock-t-1                                                   ; Stock availible in t-1

  ]

Banks-own [
  Retail-Bank?                                                 ; Banks property, true if it is a retail bank, false it is a wholesaler bank
  Equity-Workers                                               ; sum of deposit at bank of the households
  Equity-Capitalists                                           ; sum of deposit at bank of the capitalists
  Equity-C-Firms                                               ; sum of deposit at bank of the C-Firms
  Equity-K-Firms                                               ; sum of deposit at bank of the K-Firms
  Equity-Bank                                                  ; bank own equity
  Equity-Total                                                 ; sum of all equities
  Leverage-C-Firms                                             ; mean leverage of the C-Firms, every bank has their number
  Leverage-K-Firms                                             ; mean leverage of the K-Firms, every bank has their number
  Interest-Receive                                             ; sum of interest received last period
  Installment-Receive                                          ; sum of installment received last period
  Profit                                                       ; sum of interest received less the bank losses with their loans
  Intrabank-Loan                                               ; loans regestrited between banks
  Intrabank-rate                                               ; interest rate praticed in the Intrabank-Loan

  ]

;;******************************************************************************XXXXXXXX_SETUP_XXXXXXXX******************************************************************************

to Setup
  clear-all
  if Workers?                                       [ Setup-Workers    ]             ; call the setup for workers
  if C-Firm?                                        [ Setup-C-Firms    ]             ; call the setup for C-Firms
  if K-Firm?                                        [ Setup-K-Firms    ]             ; call the setup for K-Firms
  if Capitalist?                                    [ Setup-Capitalist ]             ; call the setup for Capitalists
  Setup-Globals                                                                      ; call the setup for globals
  if Bank?                                          [ Setup-Bank       ]             ; call the setup for banks
  reset-ticks                                                                        ; netlogo needs
end

to Setup-Globals
  set Free-Interest-Rate             natural-interest-rate
  set GDP                            0                                               ; initial GDP
  set Unemployment                   0                                               ; initial Unemployment rate
  set Consumption                    0                                               ; initial value of consumption
  set Total-Investment               0                                               ; initial value of total investment
  set Goods-Price-Level              2                                               ; initial goods price level
  set Capital-Price-Level            2                                               ; initial capital goods price level
  set Total-Loans                    0                                               ; initial value for total loans loans
  set Total-Loans-C-Firms            0                                               ; initial value for c-loans
  set Total-Loans-K-Firms            0                                               ; initial value for K-Loans
  set C-Firms                        Firms      with [    C-Firms?]                  ; building the group with all C-Firms
  set K-Firms                        Firms      with [not C-Firms?]                  ; building the group with all K-firms
  set Capitalists                    Households with [not Worker? ]                  ; building the group with all Capitalist
  set Workers                        Households with [    Worker? ]                  ; building the group with all workers
  set Matrix-C-Firms-Bankruptcy      matrix:make-constant  2 Window 0                ; building the initial matrix which will be used by R-Netlogo later
  set Matrix-K-Firms-Bankruptcy      matrix:make-constant  2 Window 0                ; building the initial matrix which will be used by R-Netlogo late
  set Mj                             0                                               ; helper with the matrix above
  set Mjk                            0                                               ; helper with the matrix above
  if not Random-Seed?               [ random-seed 12345 ]                            ; random seed may or not be used

end

to Setup-Workers
  create-Households Number-of-workers [
    setxy                            random-xcor random-ycor                         ; the initial position of the worker is random
    set shape                        "person"                                        ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set size                         0.50                                            ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set color                        blue                                            ; just for fun, not necessary for the model, used only for display in HUD netlogo
    set Worker?                      true                                            ; proper of the household, true indicates it is a worker
    set Human-Wealth                 1                                               ; initial value for Human Wealth, utilized to calculate the consume process
    set Current-Income               0                                               ; intitial income
    set Deposit-Bank                 Initial-households-personal-assets              ; initial value of deposit at bank
    set Desire-Consume               0                                               ; initial value for desire consume
    set Consume                      0                                               ; initial value for consume
    set Price-List                   [ ]                                             ; setup for the list which will be use to colect the price of the C-Firms visited
    ]
end

to Setup-Capitalist
  ask n-of 250 patches [                                                             ; each patches represent one firm, this command ensure will be one capitalist for each firm
    sprout-Households 1 [                                                            ; ask to create one capitalist in each position
      set shape                        "person"                                      ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set size                         0.50                                          ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set color                        white                                         ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set Worker?                      false                                         ; proper of the household, false means it is a capitalist
      set Human-Wealth                 1                                             ; initial value for human wealth
      set Current-Income               0                                             ; initial value for current income
      set Deposit-Bank                 Initial-households-personal-assets            ; initial value for deposit at bank
      set Desire-Consume               0                                             ; initial value for desire-consume
      set Consume                      0                                             ; initial value of consume
      set Price-List                   [ ]                                           ; setup for the list which will be use to colect the price of the C-Firms visited
      create-Owner-from                one-of Firms-Here [set hidden? true]          ; this create a link between the firm and the capitalist. Thus, one will know eachother for all the porpouse, dividen, rebuild a broken firm, etc.
      ]
    ]
end

to Setup-C-Firms
  ask n-of 200 patches with [pycor < 8] [                                             ; this command ask the 200 patches which are position at grid with y coordinates less than 8
    sprout-Firms 1 [                                                                  ; ask to create one firm at this patche
      set shape                      "house"                                          ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set size                       0.75                                             ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set color                      orange                                           ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set C-Firms?                   true                                             ; proper of the Firms, true means it is a consumption producer firm
      set Price                      2                                                ; initial value for price
      set Capital                    Initial-capital                                  ; initial value for capital
      set Stock                      Initial-production-C-firms                       ; initial value for stock
      set Production                 Initial-production-C-firms                       ; initial value for production
      set Revenue                    0                                                ; initial value for revenue
      set Salles                     0                                                ; initial value for sales
      set Need-Cash                  0                                                ; initial value for cash need
      set Desire-Investment          0                                                ; initial value for desire investment
      set Investment                 0                                                ; initial value for investment
      set Number-Employee            0                                                ; initial value for number of employee
      set Desire-Employee            0                                                ; initial value for desire employees
      set Price-List                 [ ]                                              ; setup for the list which be used to colect the prices from the K-firms visited trying to buy capital
      set Vacancy                    0                                                ; initial value for vacancy
      set Bank-Deposit               Initial-liquidity-of-Firms                       ; initial money deposited at the bank
      set Investment-Memory          Initial-Capital                                  ; initial value for Investment-Memory; parameter utilized to calculate the desire investment
      set Util-Capacity              0                                                ; initial value for util-capacity
      ]
    ]
end

to Setup-K-Firms
    ask n-of 50 patches with [pycor >= 8] [                                           ; this command ask 50 patches which are located at coordinates y >= 8
    sprout-Firms 1 [                                                                  ; ask to create one firm in each of those patches
      set shape                      "house"                                          ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set size                       0.75                                             ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set color                      brown                                            ; just for fun, not necessary for the model, used only for display in HUD netlogo
      set C-Firms?                   false                                            ; proper of Firms, false mean it is a capital producer firm
      set Price                      2                                                ; initial value for price
      set Capital                    Initial-capital                                  ; initial value for capital
      set Stock                      Initial-production-K-firms                       ; initial value for stock
      set Production                 Initial-production-K-firms                       ; initial value for production
      set Revenue                    0                                                ; initial value for revenue
      set Salles                     0                                                ; initial value for sales
      set Need-Cash                  0                                                ; initial value for cash needed
      set Desire-Investment          0                                                ; initial value for desire investment
      set Investment                 ""                                               ; initial value for investment
      set Number-Employee            0                                                ; initial number for employee
      set Desire-Employee            0                                                ; initial value for desire employee
      set Price-List                 ""                                               ; not applied to K-Firms
      set Vacancy                    0                                                ; initial value for vacancy
      set Bank-Deposit               Initial-liquidity-of-Firms                       ; initial money deposited at the bank
      set Investment-Memory          ""                                               ; not applied to K-Firms
      set Util-Capacity              ""                                               ; not applied to K-Firms
      ]
    ]
end

to Setup-Bank
  create-Banks 1 [                                                                    ; create one bank
    set Retail-Bank?                 true                                             ; bank proper, true means it is a retail bank type
    set Equity-Workers               sum [deposit-bank] of workers                    ; explicit
    set Equity-Capitalists           sum [deposit-bank] of capitalists                ; explicit
    set Equity-C-Firms               sum [bank-deposit] of c-firms                    ; explicit
    set Equity-K-Firms               sum [bank-deposit] of k-firms                    ; explicit
    set Equity-Bank                  Initial-equity-of-the-bank                       ; initial value fo the own bank equity
    set Equity-Total                 Equity-Workers + Equity-Capitalists + Equity-C-Firms + Equity-K-Firms + Equity-Bank
    set Intrabank-loan               0                                                ; initial value for the intrabank loans
    set Intrabank-Rate               free-interest-rate                            ; initial intrabank interest rate
    ]
  if Wholesales-Banks? [                                                              ; swith positioned in the netlogo HUD
    create-Banks 1 [                                                                  ; ask to create one bank
      set Retail-Bank?               false                                            ; bank proper, false means it is a wholesaler bank type
      set Equity-Workers             0                                                ; only retails bank had this
      set Equity-Capitalists         0                                                ; only retails bank had this
      set Equity-C-Firms             0                                                ; only retails bank had this
      set Equity-K-Firms             0                                                ; only retails bank had this
      set Equity-Bank                Initial-equity-of-the-bank                       ; initial value of bank own equity
      set Equity-Total               Equity-Bank                                      ; initial value for total equity
      set Intrabank-loan             0                                                ; initial value for intrabank loans
      set Intrabank-Rate             free-interest-rate                            ; initial value for intrabank-rate
      ]
    ]
  set Retail-Banks                   Banks with [    Retail-Bank?]                    ; create a group with all retails banks
  set Wholesale-Banks                Banks with [not Retail-Bank?]                    ; create a group with all wholesalers banks
end

;;******************************************************************************XXXXXXXX_GO_XXXXXXX******************************************************************************

to Go                                                                                 ; here will be calling the functions
  Function-Job-Market                                                                          ; this function plays the job market process
  Function-Consumption-Market                                                                  ; this function plays the consumption market process
  Function-Capital-Market                                                                      ; this function plays the capital market process
  Function-UpDate-Production                                                                   ; this function plays the production process
  Function-UpDate-Globals                                                                      ; this function update the globals
  Function-Cash-Flows                                                                          ; this function update the cash flow
  Function-Credit-Market                                                                       ; this function update the credit market
  if Central-Bank? [ Function-Call-Central-Bank ]

  tick                                                                                ; netlogo needs
end



;;--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
;;******************************************************************************XXXXXXXX_Functions_XXXXXXX******************************************************************************
;;--------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------



;;******************************************************************************XXXXXXXX_Function-JobMarket_XXXXXXX******************************************************************************



To Function-Job-Market                                                                                                              ; here will be called the subfunctions
  ask Firms                                 [ SubF-UpDate-Desire-Workers set Need-Cash 0                    ]                   ; reset the firms need for cash and update the desire workers number
  ask Workers with [current-Income = 0]     [ SubF-Find-a-Job                                               ]                   ; ask workers which did not have a job to look for one
  ask Workers                               [ set Deposit-Bank   Deposit-Bank + Current-Income         ]                   ; ask workers to update their income
  ask Firms                                 [ set Bank-Deposit   Bank-Deposit - Number-Employee * wage ]                   ; ask firms to updated their bank account
end

to SubF-UpDate-Desire-Workers                                                                                                   ; this subfunction update the desire workers
  if  C-Firms?                              [                                                                              ; if C-firms is true, only C-Firm run this
     set Desire-Employee   min list Desire-Employee ((Capital * Productivity-of-capital) / Productivity-of-labor)          ; set desire employee considering the capital production restriction
     if C-Firm_Budget_rest02? [ if Desire-Employee * wage > bank-deposit [                                             ; update the desire employee considering the money availible
         set Desire-Employee   max list 0 (int (Bank-Deposit / wage) + 1)                                                  ; update the desire employee considering the money availible
          ] ]
     ]
  set Vacancy                                Desire-Employee   -   Number-Employee                                         ; opening new vacancies to contract more employee
  while                                     [Vacancy < 0 and Number-Employee > 0] [                                        ; if vacancy < 0 need to fire some one
    ask one-of my-in-Hireds [                                                                                              ; choose one employee
      ask other-end [                                                                                                      ; ask this employee to set...
        set Current-Income                   0                                                                             ; their income as 0, it was fired
        ]
      die                                                                                                                  ; die means this worker no more has a contract with the firm, fired
      ]
    set Vacancy                              Vacancy           + 1                                                         ; confirm the fired process, perhaps need to fire another one
    set Number-Employee                      Number-Employee   - 1                                                         ; update the number of employee after the firing process
    ]
end

to SubF-Find-a-Job                                                                                                              ; this subfunction is used by the worker trying to find a job position
  repeat Number-of-firms-visited-by-unemployed-workers [                                                                   ; the worker can try a maximum number of times to get a job, if not find he still unemployeed
    setxy                                    random-xcor random-ycor                                                       ; this means the worker visit a firm at the grid
    let Vacancy?                             false                                                                         ; auxiliar variable to this subfunction
    ask one-of Firms-here  [                                                                                               ; ask to firm visited if ...
      if Vacancy > 0 [                                                                                                     ; ... it has an open position
        set Vacancy?                         true                                                                          ; update auxiliar variable
        set Number-Employee                  Number-Employee + 1                                                           ; update the number of employee
        set Vacancy                          Vacancy         - 1                                                           ; update the vacancy of this firm
      ]
    ]
    if Vacancy? [                                                                                                          ; using the auxiliar variable to update the worker status
      set Current-Income                      Wage                                                                         ; update the current-income of the worker
      create-Hired-to                         one-of Firms-here [set hidden? true]                                         ; this link means the worker has a job, and it and the firm knows abouot eachother
      stop                                                                                                                 ; for the repeat function, if it finds a job did not need try once more
    ]
  ]
end




;;******************************************************************************XXXXXXXX_Function-ConsumpMarket_XXXXXXX******************************************************************************




to Function-Consumption-Market                                                                                                      ; here will be called the subfunctions of the consumption market
  ask C-firms    [ set demand 0 set salles 0 set revenue 0]                                                                ; updating properties of the C-Firms
  ask Households [                                                                                                         ; explicit
    SubF-UpDate-Desire-Consume                                                                                                  ; call subfunction for households update their consume desire
    SubF-Visit-Firms                                                                                                            ; call subfunction for households visit firm trying to consume
    if Desire-Consume > 0 [ SubF-Try-Consume ]                                                                                  ; if wana consume something, try buy it
    ]
end

to SubF-UpDate-Desire-Consume
  set Human-Wealth                           (Memory-parameter-human-wealth             * Human-Wealth) + (1 - Memory-parameter-human-wealth) * Current-Income                 ; update human wealth
  set Desire-Consume                          min list    Deposit-Bank                  ((Fraction-of-wealth-devoted-to-consumption * Deposit-Bank) + Human-Wealth)            ; update desire consume, it can not be higher than the money the household has
  set Desire-Consume                          max list  0 Desire-Consume                                                                                                       ; ensure desire consume is not negative
end

to SubF-Visit-Firms                                                                                                             ; this subfunction rule the shopping process, visitng the firms
  let Who01                                   ""                                                                           ; auxiliar variable, save the name of the first firm visited
  let Who02                                   ""                                                                           ; auxiliar variable, save the name of the second firm visited
  let Price01                                 ""                                                                           ; auxiliar variable, save the price of the firm visited
  let Price02                                 ""                                                                           ; auxiliar variable, save the price of the second firm visited
  setxy random-xcor random 8                                                                                               ; choose the address of one random C-firm
  ask one-of Firms-here [
    set Who01                                 Who                                                                          ; save the name of this firm
    set Price01                               Price                                                                        ; save the price of this firm
    ]
  setxy random-xcor random 8                                                                                               ; choose the address of the second firm
  ask one-of Firms-here [
    set Who02                                 Who                                                                          ; save the name of this firm
    set Price02                               Price                                                                        ; save the price of this firm
    ]
  while [Who01 = Who02] [                                                                                                  ; if happened to visited the same firm twice, find another one, repeat the process above
    setxy random-xcor random 8
    ask one-of Firms-here [
      set Who02                                 Who
      set Price02                               Price
      ]
    ]
  ifelse Price01 < Price02 [set Price-List list (Firm Who01) (Firm Who02) ] [set Price-List list (Firm Who02) (Firm Who01) ] ; create an organized list, the firm with the lowest price comes first
end

to SubF-Try-Consume
  let Firm-A                                    first Price-List                                                          ; extract the name of the firm with a better price
  let Firm-B                                    last  Price-List                                                          ; extract the name of the firm with the worst price
  let Qtt                                       Desire-Consume / [Price] of Firm-A                                        ; auxiliar variable, record the quantity can be bought
  let expenses                                  0                                                                         ; auxiliar variable to update the deposit at bank later
  let cons                                      0                                                                         ; auxiliar variable to record the quantity consumed
  ask Firm-A [set Demand                        Demand + Qtt ]                                                            ; update the demand of C-firms
  if [Stock] of Firm-A < Qtt [                                                                                            ; if there is not availible goods at the first firm ...
    ask Firm-B [set Demand                      Demand + Qtt - [Stock] of Firm-A ]                                        ; also the second firm has a demand for their product
    ]
  ask Firm-A [                                                                                                            ; trying to consume at the better price
    ifelse Stock > Qtt [                                                                                                  ; first situation, better one which can consume all at the better price
      set Stock                                 Stock  - Qtt                                                              ; updated the stock of the firm
      set Salles                                Salles + Qtt                                                              ; update the sales
      set Revenue                               Price  * Qtt  + Revenue                                                   ; update the revenue
      set expenses                              Price  * Qtt  + expenses                                                  ; update the auxiliar variable, used later to update the household bank account
      set cons                                  cons   + Qtt                                                              ; update the auxiliar variable to register the qtt consumed
      set Qtt                                   0                                                                         ; means no need to bought anymore
      ] [
      if Stock > 0 [                                                                                                      ; second situation, buying only part at first firm
        set Salles                              Salles  + Stock                                                           ; update sales
        set Revenue                             Price   * Stock + Revenue                                                 ; update revenue
        set expenses                            Price   * Stock + expenses                                                ; update auxiliar variable, later used to update the bank account of the household
        set cons                                cons    + Stock                                                           ; update stock
        set Qtt                                 Qtt     - Stock                                                           ; update auxiliar variable to register the qtt consumed
        set Stock                               0                                                                         ; update stock
        ]
      ]
    ]; end ask Firm-A
  if Qtt > 0 [                                                                                                            ; means household did not consume all it wanted to in the first firm
    set Qtt                                     (Qtt * [Price] of Firm-A) / [Price] of Firm-B                             ; auxiliar variable, max quantity can consume at this price
    ask Firm-B [                                                                                                          ; updating values at the second firm
      ifelse Stock > Qtt [                                                                                                ; as above...
        ;first situation, buy all
        set Stock                                 Stock  - Qtt
        set Salles                                Salles + Qtt
        set Revenue                               Price  * Qtt  + Revenue
        set expenses                              Price  * Qtt  + expenses
        set cons                                  cons   + Qtt
        set Qtt                                   0
        ] [
        ;second situation, did not consume all he wants
        if Stock > 0 [
          set Salles                              Salles  + Stock
          set Revenue                             Price   * Stock + Revenue
          set expenses                            Price   * Stock + expenses
          set cons                                cons    + Stock
          set Qtt                                 Qtt     - Stock
          set Stock                               0
          ]
        ]
      ];end ask firm-B
    ];end Qtt > 0
  ;update households atributes
  set Consume                                     cons                                                                  ; update the Consume of the households
  set Deposit-Bank                                Deposit-Bank - expenses                                               ; update the bank account of the households

end




;;*********************************************************************************************Function-Capital-Market*********************************************************************************************




to Function-Capital-Market                                                                                              ; this function set the rules for the capital goods market
  ask K-Firms [ set Demand 0 set revenue 0 set Salles  0 ]                                                              ; upadte K-Firms properties
  ask C-Firms [                                                                                                         ; updating the c-Firms properties
    set Investment                 0
    SubF-UpDate-Investment-Desire                                                                                       ; call sub-function to update investment
    if Desire-Investment > 0 [ SubF-UpDate-Investment ]                                                                 ; if there is need for investment call the sub-function for investment
    ]
end

to SubF-UpDate-Investment-Desire                                                                                        ; sub-function with procedure to update the investment need of a C-Firm
  set Desire-Investment            0                                                                                    ; reset variable
  let rand                         random-float 1                                                                       ; auxiliar sub-function variable
  if  rand < Probability-of-investing [                                                                                 ; check if it is time to invest
    if ticks > 1 [                                                                                                      ; assure that there is no investment in time equal zero
      set Investment-Memory        (Memory-parameter-investment * Investment-Memory) + (1 - Memory-parameter-investment) * Util-Capacity * Capital                               ; begin of equation of capital motion
      set Desire-Investment        max list 0 (( (1 / Desire-capacity-utilization-rate) + (Depreciation-of-capital / Probability-of-investing) ) * Investment-Memory - Capital)  ; end of equation of capital motion
      set Need-Cash                Need-Cash   +  Desire-Investment * Capital-Price-Level                                                                                        ; check for cash needed
      ]
    ]
end

to SubF-UpDate-Investment                                                                                               ; this subfunction is called when there is need for investment
  if Desire-Investment > 0 [                                                                                            ; just trying to find product in two diferents K-Firms
    SubF-Visit-K-Firms                                                                                                  ; visit two k-firms, take note of their prices
    SubF-Buy-Capital                                                                                                    ; sub-function used to bought capital
  ]
end

to SubF-Visit-K-Firms                                                                                                   ; searching for the better price, similar to the households sub-functions, check for details there
  let X                                xcor                                                                             ; the diference is that the firm need to have a fixed position. this X and Y auxiliar variable save the firm position at the grid
  let Y                                ycor
  let Who01                            ""
  let Price01                          ""
  let Who02                            ""
  let Price02                          ""
  setxy                                random-xcor (random 2 + 8)
  ask one-of other Firms-here [
    set Who01                          Who
    set Price01                        Price
    ]
  setxy                                random-xcor (random 2 + 8)
  ask one-of other Firms-here [
    set Who02                          Who
    set Price02                        Price
    ]
  while [ Who01 = Who02 ] [
    setxy                             random-xcor (random 2 + 8)
    ask one-of other Firms-here [
      set Who02                       Who
      set Price02                     Price
    ]
    ]
  ifelse Price01 < Price02 [set Price-List list (Firm Who01) (Firm Who02) ] [set Price-List list (Firm Who02) (Firm Who01) ]
  setxy                              X Y                                                                                  ; return the firm to their position after the prices search
end

to SubF-Buy-Capital                                                                                                       ; sub-function to buy capital goods
  let expenses                                  0                                                                         ; sub-function auxiliar variable to save the expenses of the C-Firm
  let New-Inv                                   0                                                                         ; sub-function auxiliar variable to save the qtt capital bough by the C-Firm
  let Firm-A                                    first Price-List                                                          ; extract the name of the K-firm with the better price
  let Firm-B                                    last  Price-List                                                          ; extract the name of the K-firm with the seccond price
  let Qtt                                       Desire-Investment                                                         ; auxiliar variable register the qtt C-firm want to buy
  if c-Firm_Budget_rest01? [                                                                                              ; including budget restriction, this way the demand of k-firm is afected by the monetary restriction
    let Budget                                    Qtt * [Price] of Firm-A                                                 ; auxiliar variable with the monetary value the C-Firm can afford to invest
    if Budget > Bank-Deposit [                                                                                            ; check if the C-Firm has enough money to invest
      set Qtt                                     max list 0 (Bank-Deposit / [Price] of Firm-A)                           ; rebalance the investment according to money availible
    ]
  ]
  ask Firm-A [ set Demand                       Demand + Qtt ]                                                            ; updating demand at K-Firm
  if [stock] of Firm-A < Qtt [                                                                                            ; check if the K-Firm has enough stock to supply
    ask Firm-B [ set Demand                     Demand + Qtt - [Stock] of Firm-A ]                                        ; with the first did no has enough stock, also update the demand of the second firm
    ]
  ifelse [stock] of Firm-A >= Qtt [                                                                                       ; trying to buy
    ask Firm-A [                                                                                                          ; first situation, better one, it is possible to bought all at the better price
      set stock                                 stock   - Qtt                                                             ; updating properties of the K-Firm ...
      set Revenue                               Price   * Qtt + Revenue                                                   ; ...
      set expenses                              Price   * Qtt + expenses                                                  ; ...
      set Salles                                Salles  + Qtt                                                             ; ...
      set New-Inv                               New-Inv + Qtt                                                             ; ...
      set Qtt                                   0                                                                         ; ...     end first situation for Firm-A
      ]
     ] [
     if [stock] of Firm-A > 0 [                                                                                           ; second situation, there is not enough stock at the first firm
       ask Firm-A [                                                                                                       ; updating properties of the K-Firm ...
         set Revenue                            Price   * Stock + Revenue                                                 ; ...
         set expenses                           Price   * Stock + expenses                                                ; ...
         set Salles                             Salles  + Stock                                                           ; ...
         set Qtt                                Qtt     - Stock                                                           ; ...
         set New-Inv                            New-Inv + Stock                                                           ; ...
         set Stock                              0                                                                         ; ...        end second situation for Firm-A
         ]
       ]
     ]

  if Qtt > 0 [                                                                                                            ; means the C-Firm did not consume all investment needed
    ifelse [stock] of Firm-B >= Qtt [                                                                                     ; first situation, it is possible to bought all it is needed
      ask Firm-B [                                                                                                        ; updating properties of the K-Firm ...
        set stock                                 stock   - Qtt                                                           ; ...
        set Revenue                               Price   * Qtt + Revenue                                                 ; ...
        set expenses                              Price   * Qtt + expenses                                                ; ...
        set Salles                                Salles  + Qtt                                                           ; ...
        set New-Inv                               New-Inv + Qtt                                                           ; ...
        set Qtt                                   0                                                                       ; ...      end first situation for Firm-B
      ]
    ] [
    if [stock] of Firm-B > 0 [                                                                                            ; second situation, the C-Firm will not invest all it wants to
      ask Firm-B [                                                                                                        ; updating the properties of the K-Firm ...
        set Revenue                            Price   * Stock + Revenue                                                  ; ...
        set expenses                           Price   * Stock + expenses                                                 ; ...
        set Salles                             Salles  + Stock                                                            ; ...
        set Qtt                                Qtt     - Stock                                                            ; ...
        set New-Inv                            New-Inv + Stock                                                            ; ...
        set Stock                              0                                                                          ; ...      end second situation   Firm-B
      ]
    ]
    ]
  ]
    set Investment                              New-Inv                                                                   ; updating the propertie of the C-Firm, the investment it was acomplished
    set Bank-Deposit                            Bank-Deposit - expenses                                                   ; updating the bank account the C-Firm
end




;****************************************************************************************Function-UpDate-Production*********************************************************************************************




to Function-UpDate-Production                                                                                            ; this function update the production of Firms
  ask K-Firms [                                                                                                          ; updating the production of K-Firms
    set Production                  Number-Employee         * Productivity-of-labor                                      ; updating actual production
    set Stock                       Stock                   * (1 - depreciation-of-capital)           + Production  ; updating stock
    set Bank-Deposit                Bank-Deposit            + Revenue                                                    ; update bank account
    set Delta                       Production              - Demand                                                     ; variable used to decide new production
    ifelse K-Qtt-Function?          [ set Desire-Employee    Report-K-Update-Desire-Employee             ]               ; the user may force the k-firm to produce capital goods ...
                                    [ set Desire-Employee    Demand   /   Productivity-of-labor   ]                      ; ... if the switcher at HUD is turned off
    ifelse K-UpDate-Price?          [ set Price              Report-K-Update-Price                       ]               ; the user may force the capital price be fixed if ...
                                    [ set Price              wage / productivity-of-labor         ]                      ; ... the switcher at HUD is turned off
    ]
  ask C-Firms [                                                                                                          ; updating the production of C-Firms
    set Capital                     Capital                   - Capital * Util-Capacity * Depreciation-of-capital + Investment                      ; update the capital avalible for production process
    set Production                  min list                 (Number-Employee   *   Productivity-of-labor) (Capital * Productivity-of-capital)      ; update actual production limited by the availible capital
    set Stock                       Production                                                                           ; the C-Goods are perishable, so the actual stock are equal the production
    set Util-Capacity               Production               / (Capital * Productivity-of-capital)                       ; updat utilized capacity
    set Bank-Deposit                Bank-Deposit             + Revenue                                                   ; update bank account
    set Delta                       Production               - Demand                                                    ; variable used to decide new production level
    ifelse C-Qtt-Function?          [ set Desire-Employee    Report-C-Update-Desire-Employee ]                           ; the user may force the C-Firm to produce C-goods ...
                                    [ set Desire-Employee    Demand            /   Productivity-of-labor   ]             ; ... if the switcher at HUD is turned off
    ifelse C-UpDate-Price?          [ set Price              Report-C-Update-Price               ]                       ; the user may force the consumption good be fixed ...
                                    [ set Price              wage / productivity-of-labor ]                              ; ... if the switcher at HUD is turned off
    ]
end

to-report Report-C-Update-Price                                                                                                             ; reporting the new price for C-Firms ...
  let N-P                           Price                                                                                                   ; local auxialiar variable
  if  (delta <= 0) and              (Price <  Goods-Price-Level)  [ set N-P      Price * (1 + random-float 1 * Price-adjust-parameter)  ]   ; rule to reduce current price
  if  (delta >  0) and              (Price >  Goods-Price-Level)  [ set N-P      Price * (1 - random-float 1 * Price-adjust-parameter)  ]   ; rule to increase current price
  report N-P                                                                                                                                ; report new price
end

to-report Report-K-Update-Price                                                                                                             ; reporting the new price for K-firms
  let N-P                           Price                                                                                                   ; local auxiliar variable
  if (delta <= 0) and               (Price <   Capital-Price-Level) [ set N-P Price * (1 + random-float 1 * Price-adjust-parameter )  ]     ; rule to reduce price
  if (delta >  0) and               (Price >   Capital-Price-Level) [ set N-P Price * (1 - random-float 1 * Price-adjust-parameter )  ]     ; rule to increase price
  report                            N-P                                                                                                     ; report new price
end

to-report Report-C-Update-Desire-Employee                                                                                                                            ; reporting new need for a C-firm employees
  let N-E                           Number-Employee                                                                                                                  ; local auxiliar variable
  if (delta <= 0) and               (Price >= Goods-Price-Level)  [ set N-E ((Production - Quantity-adjust-parameter * delta) / productivity-of-labor) ]             ; possible condition to change actual number of employee
  if (delta >  0) and               (Price <  Goods-Price-Level)  [ set N-E ((Production - Quantity-adjust-parameter * delta) / productivity-of-labor) ]             ; other possible condition to change actual number of employee
  report N-E                                                                                                                                                         ; report new desire employee
end

to-report Report-K-Update-Desire-Employee                                                                                                                             ; reporting new need for a C-firm employees
  let N-E                           Number-Employee                                                                                                                   ; local auxiliar variable
  if (delta <= 0) and               (Price >=  Capital-Price-Level) [ set N-E ((Production - Quantity-adjust-parameter * delta - Stock) / productivity-of-labor)  ]   ; possible condition to change actual number of employee
  if (delta >  0) and               (Price <   Capital-Price-Level) [ set N-E ((Production - Quantity-adjust-parameter * delta - Stock) / productivity-of-labor)  ]   ; other possible contidtion to change actual number of employee
  report N-E                                                                                                                                                          ; report new need for employees
end




;;***************************************************************************************************GLOBALS*********************************************************************************************




to Function-UpDate-Globals                                                                                                                                    ; this function update the globals
  set Unemployment              1 - ((sum [Number-Employee]              of firms) / Number-of-workers )                                                      ; update the unemployemnt rate
  set GDP                             sum [Production  ]                 of Firms - sum [stock]  of K-Firms   if GDP < 0 [set GDP 1]                          ; update the GDP, sum production - stocks
  set Total-Investment                sum [Investment  ]                 of C-Firms                                                                           ; update investment realized by the C-Firms
  set Consumption                     sum [Consume     ]                 of Households                                                                        ; update the consumption
  set Total-Loans                     sum [Current-Retail-Loan]          of Firms     +     sum [Current-wholesale-Loan]          of Firms                    ; update the sum of retail loans, all firms
  set Total-Loans-C-Firms             sum [Current-Retail-Loan]          of C-Firms   +     sum [Current-wholesale-Loan]          of C-Firms                  ; update the sum of retail loans for C-Firms
  set Total-Loans-K-Firms             sum [Current-Retail-Loan]          of K-Firms   +     sum [Current-wholesale-Loan]          of K-Firms                  ; update the sum of retail loans for K-Firms
  set Goods-Price-Level_t-1               Goods-Price-Level
  set Capital-Price-Level_t-1             Capital-Price-Level
  if sum [Salles] of C-Firms != 0   [ set Goods-Price-Level              sum [Revenue     ] of C-Firms                  / sum [Salles] of C-Firms ]           ; update the goods price level
  if sum [Salles] of K-Firms != 0   [ set Capital-Price-Level            sum [Revenue     ] of K-Firms                  / sum [Salles] of K-Firms ]           ; update the capital price level
end




;*********************************************************************************************Function-Cash-Flows*********************************************************************************************




to Function-Cash-Flows                                                                                                         ; this function set the moviment of cash in the system
  ask C-Firms [                                                                                                                ; Cash flow of C-Firms
    let div                                     0                                                                              ; auxiliar variable
    if  Profit? [                                                                                                              ; HUD control
      set Profit                                max list 0 ( Revenue - Number-Employee * Wage)                                 ; check if had profit this period
      set Dividend                              min list   ( Dividend-payout-ratio     * Profit) (max list 0 (Bank-Deposit))   ; if there was profit setup dividend
      set Bank-Deposit                          Bank-Deposit - Dividend                                                        ; reduce the bank account the dividen which will be payed to the capitalist
      set Div                                   Dividend                                                                       ; auxiliar variable bring the div to the capitalist later
      ]
    ask one-of my-out-owners [                                                                                                 ; calling the owner of this firm
      ask other-end [                                                                                                          ; other end is the capitalist
        set Current-Income                     div                                                                             ; update the current income
        set Deposit-Bank                       Deposit-Bank   +   div                                                          ; update the household bank account
      ]
    ]
    ]

  ask K-Firms [                                                                                                                ; cash flow of the K-firms
    let div                                     0                                                                              ; auxiliar variable
    if  Profit? [                                                                                                              ; HUD control
      set Profit                                max list 0 ( Revenue - Number-Employee * Wage)                                 ; check if had profit this period
      if Prof-K-Firms? [                                                                                                       ; switch in HUD control
        set Profit                              max list 0 ( Bank-Deposit )                                                    ; set max profit to avoid negative bank account
         ]
      set Dividend                              min list   ( Dividend-payout-ratio     * Profit) (max list 0 (Bank-Deposit))   ; set dividend
      set Bank-Deposit                          Bank-Deposit - Dividend                                                        ; update bank account
      set Div                                   Dividend                                                                       ; auxiliar variable bring the value of div to the capitatilist later
      ]
    ask one-of my-out-owners [                                                                                                 ; calling the owner of this firm
      ask other-end [                                                                                                          ; other is the capitalist
        set Current-Income                     div                                                                             ; update the current income
        set Deposit-Bank                       Deposit-Bank   +   div                                                          ; update the household bank account
      ]
    ]
    ]

  ask Retail-Banks [                                                                                                           ; update the data into the retail bank system
    set Equity-Workers               sum [deposit-bank] of workers                                                             ; explicit
    set Equity-Capitalists           sum [deposit-bank] of capitalists                                                         ; explicit
    set Equity-C-Firms               sum [bank-deposit] of c-firms                                                             ; explicit
    set Equity-K-Firms               sum [bank-deposit] of k-firms                                                             ; explicit
    set Equity-Bank                  Initial-equity-of-the-bank                                                                ; explicit
    set Equity-Total                 Equity-Workers + Equity-Capitalists + Equity-C-Firms + Equity-K-Firms + Equity-Bank       ; explicit
    ]
end




;;******************************************************************************************Function-Credit-Market*********************************************************************************************




to Function-Credit-Market                                                                             ; this function rules the credit market system
  SubF-Call-Matrix-Formation                                                                          ; function to set the matrix used in R to calculate the life spam of firms
  set C-Broke                         0                                                               ; global variable for total C-firm went broke this period
  set K-Broke                         0                                                               ; global variable for total K-firm went broke this period
  if Wholesales-Banks? [ SubF-Call-Wholesales  SubF-Call-Intrabank ]                                  ; switch in HUD to control the introduction of the wholesaler bank
  if Retail-Banks? [                                                                                  ; switch in HUD to controle the retail bank
    ask Firms [                                                                                       ; firms update their status with the retail bank
      SubF-Update-Installment                                                                         ; to update the installment
      SubF-Update-Cash-Need                                                                           ; to update firm cash need
      SubF-Update-Assets                                                                              ; to update assets value
      SubF-Update-Leverage                                                                            ; to update the leverage, all this is used to rule interest rate praticed later
    ]
    ask K-firms [
      SubF-Ask-for-K-Credit                                                                           ; Check if this K-Firm needs credit, if it is need will be asked in this subfunction
      ]
    ask C-Firms [
      SubF-Ask-for-C-Credit                                                                           ; check if this C-firm needs credti, if it is need will be asked in this subfunction
      ]
    let C-Loans                       sum [Current-Retail-Loan] of Firms                              ; local variable
    let I-Receive                     sum [Interest-Pay       ] of Firms                              ; local variable
    let D-Receive                     sum [Installment-Pay    ] of Firms                              ; local variable
    ask Retail-Banks [                                                                                ; the retail bank update the balance after the new was created
      set Total-Loans-C-Firms         sum [Current-Retail-Loan] of C-Firms                            ; explicit
      set Total-Loans-K-Firms         sum [Current-Retail-Loan] of K-Firms                            ; explicit
      set Total-Loans                 sum [Current-Retail-Loan] of Firms                              ; explicit
      set Interest-Receive            I-Receive                                                       ; explicit
      set Profit                      Profit + I-Receive                                              ; explicit
      set Installment-Receive         D-Receive                                                       ; explicit
    ]

    if C-Broke? [                                                                                     ; switch check if the user want the broke procedure
      ask C-Firms [                                                                                   ; this rules is for C-firm broke
        ifelse (Retail-Bankruptcy > T-Ban) or (Wholesale-Bankruptcy > T-Ban) [                        ; check if this firm did not pay their debts with a bank
          SubF-Go-Bankruptcy-Type-C                                                                   ; subfunction to lead the broke process
          set Broke 1                                                                                 ; if this bad increase the broke status
          ] [ set Broke 0 ]                                                                           ; if the firm has no open debt reset their broke status
         ]
      ]
    if K-Broke? [                                                                                     ; the same as the C-firm, check above for details
      ask K-Firms [
        ifelse (Retail-Bankruptcy > T-Ban) or (Wholesale-Bankruptcy > T-Ban) [
          SubF-Go-Bankruptcy-Type-K
          set Broke 1
          ] [ set Broke 0 ]
        ]
      ]
  ]
end



to SubF-Go-Bankruptcy-Type-K                                                                           ; this subfunction rules the process for how a K-firm broke
  set K-Broke                          K-Broke  + 1                                                    ; increase the global parameter which count of broke status
  set Retail-Bankruptcy                0                                                               ; reset this parameter
  set Wholesale-Bankruptcy             0                                                               ; reset the parameter
  let debts                            Current-Retail-Loan + Current-Wholesale-Loan                    ; local auxialiar variable
  ask one-of my-out-owners [                                                                           ; use the capitalist to rebuild the firm
    ask other-end [                                                                                    ; find out who is the owner of this firm
      set Deposit-Bank                 Deposit-Bank - Initial-liquidity-of-Firms                       ; update the bank account of the capitalist, reduce the moneu invested to recreat the firm
      ]
     ]
  set Current-Retail-Loan              0                                                               ; reset the loan of the firm after the bankruptcy
  set Current-Wholesale-Loan           0                                                               ; reset the loan of the firm after the bankruptcy
  set Bank-Deposit                     Initial-liquidity-of-Firms                                      ; iniciate the firm with the money from the capitalist whom owner the firm
end

to SubF-Go-Bankruptcy-Type-C                                                                           ; this subfunction rules the process for the C-firm bankruptcy
  set C-Broke                           C-Broke + 1                                                    ; increase the global parameter which count the broke status
  set Retail-Bankruptcy                 0                                                              ; reset the parameter
  set Wholesale-Bankruptcy              0                                                              ; reset the parameter
  let debts                             Current-Retail-Loan + Current-Wholesale-Loan                   ; local auxiliar variable
  ask one-of my-out-owners [                                                                           ; find out who is the owner of this firm
    ask other-end [                                                                                    ; the owner
      set Deposit-Bank                  Deposit-Bank - Initial-liquidity-of-Firms                      ; update the bank account of the capitalist, reduce the money used to rebuild the firm
      ]
     ]
  set Current-Retail-Loan               0                                                              ; reset the parameter
  set Current-Wholesale-loan            0                                                              ; reset the parameter
  if Capital-Destruction? [             set Capital Initial-Capital ]                                  ; reset the capital of the c-firm
  set Bank-Deposit                      Initial-liquidity-of-Firms                                     ; updated the bank account with the money from the capitalist
end

to SubF-Update-Installment                                                                             ; this subfunction update the installment debt
  set  Interest-Pay                     0                                                              ; reset parameter
  set  Installment-Pay                  0                                                              ; reset parameter
  if Current-Retail-Loan > 0 [                                                                         ; check if exist loan
    ifelse Bank-deposit >= ((Installment-on-debt + Interest-Retail ) * Current-Retail-Loan) [          ; check if there is money enough to pay the debts
      set Interest-Pay                Interest-Retail     * Current-Retail-Loan                        ; update the parameter with the value will be payed
      set Bank-Deposit                Bank-Deposit        - interest-pay                               ; update parameter
      set Installment-Pay             Installment-on-debt * Current-Retail-Loan                        ; update parameter
      set Bank-Deposit                Bank-Deposit        - Installment-Pay                            ; update parameter
      set Current-Retail-Loan         Current-Retail-Loan - Installment-Pay                            ; update parameter
      set Retail-Bankruptcy          0                                                                 ; as the firm pay debts this period, reset the parameter for bankruptcy
      ] [
    set Retail-Bankruptcy             Retail-Bankruptcy  +  1                                          ; if the firm did not had money she will be closer and closer to bankruptcy
      ]
    ]
end

to SubF-Update-Cash-Need                                                                               ; subfuntion used for update the need for cash
   set Need-Cash                      max list 0 (Need-Cash + Number-Employee * Wage - Bank-Deposit + (Installment-on-debt + Interest-Retail) * Current-Retail-Loan)
end

to SubF-Update-Assets                                                                                  ; subfunction to update the assets, which differ from c-firm to k-firm type
  ifelse C-Firms? [
    set Assets                       max list 0 (Stock      * Price  +  Bank-Deposit  + Capital * Capital-Price-Level)
    ] [
    set Assets                       max list 0 (Stock      * Price  +  Bank-Deposit )
    ]
end

to SubF-Ask-for-K-Credit                                                                               ; this subfunction is called when there is need for a loan
  let lev                           mean [Leverage] of K-Firms                                         ; local parameter used by a bank
  let risk                          1                                                                  ; local parameter used by a bank
  let Availible-Credit              Bank-loss-parameter * (sum [Equity-Bank] of Retail-Banks * B-Lev * ( risk  ) -  sum [Current-Retail-Loan] of Firms)  ; how much money the bank has to offer as a loan
  if (Need-Cash > 0)  and (Availible-Credit > Need-Cash) and (Retail-Bankruptcy < T-Ban) [             ; check if the need of a firm is lower then the availible credit
    set Current-Retail-Loan          Current-Retail-Loan  +  Need-Cash                                 ; updating the loans parameter
    set Bank-Deposit                 Bank-Deposit         +  Need-Cash                                 ; updating the bank account with the money received
    ifelse Update-Interest?          [ SubF-Update-Interest-Retail-K ] [ set Interest-Retail              free-interest-rate] ; call a subfunction to update the interest rate of this firm
    ]
end

to SubF-Ask-for-C-Credit                                                                               ; this subfunction is called when there is need for a loan, c-firm type
  let lev                           mean [Leverage] of C-Firms                                         ; local parameter used by a bank
  let risk                          1                                                                  ; local parameter used by a bank
  let Availible-Credit              Bank-loss-parameter * (sum [Equity-Bank] of Retail-Banks * B-Lev * ( risk  ) -  sum [Current-Retail-Loan] of Firms)  ; how much money the bank has to offer as a loan
  if (Need-Cash > 0)  and (Availible-Credit > Need-Cash) and (Retail-Bankruptcy < T-Ban) [             ; check if the need of a firm is lower then the availible credit
    set Current-Retail-Loan          Current-Retail-Loan  +  Need-Cash                                 ; updating the loans parameter
    set Bank-Deposit                 Bank-Deposit         +  Need-Cash                                 ; updating the bank account with the money received
    ifelse Update-Interest?          [ SubF-Update-Interest-Retail-C ] [ set Interest-Retail              free-interest-rate] ; call a subfunction to update the interest rate of this firm
    ]
end

to SubF-Update-Interest-Retail-C                                                                                                         ; update the interest rate of a C-firm
  let prob-die                       (1 / (1 + e ^ ( - Beta_1-C-Firms - (Beta_2-C-Firms * leverage))))                                   ; local variable, save the prob of this firm die
  set T-expected                     (1 / prob-die)                                                                                      ; calculate the life expectation of this firm
  let Numerator                      (1 + (free-interest-rate / installment-on-debt))                                                 ; local auxiliar variable, first part to calculate the new interest rate
  let Denominator                    ((1 - (1 - installment-on-debt) ^ (T-expected + 1)) / installment-on-debt)                          ; // // second part of the new interest rate
  let new-int                        Bank-gross-mark-up * ((numerator / denominator) - installment-on-debt)                              ; calculating the new interest rate
  set Interest-Retail                (Interest-Retail   * (Current-Retail-Loan - Need-Cash) + new-int * Need-Cash) / Current-Retail-Loan ; update the interest rate as mean of the new and the later interest rate
end

to SubF-Update-Interest-Retail-K                                                                                                         ; update the interest rate of a K-firm
  let prob-die                       (1 / (1 + e ^ ( - Beta_1-K-Firms - (Beta_2-K-Firms * leverage))))                                   ; the same as the C-firm, check above for details
  set T-expected                     (1 / prob-die)
  let Numerator                      (1 + (free-interest-rate / installment-on-debt))
  let Denominator                    ((1 - (1 - installment-on-debt) ^ (T-expected + 1)) / installment-on-debt)
  let new-int                        Bank-gross-mark-up * ((numerator / denominator) - installment-on-debt)
  set Interest-Retail                (Interest-Retail   * (Current-Retail-Loan - Need-Cash) + new-int * Need-Cash) / Current-Retail-Loan
end

to SubF-Update-Leverage                                                                                                                  ; updating the leverage of a firm
  if Need-Cash > 0                 [ set Leverage                      (Current-Retail-Loan + Need-Cash) / (Assets + Current-Retail-Loan + Need-Cash)]
  if (Current-Retail-Loan = 0)     [ set Leverage 0 ]
end

to SubF-Distribute-Dividends                                                                                                             ; this subfunction set the rules for dividend for banks
  ask Retail-Banks [                                                                                                                     ; the retail bank dividend
    let Div                         Profit  / (Number-of-consumptions-firms + number-of-capital-goods-firms)                             ; calculate dividend as part of the total profit divided by capitalists
    ask Capitalists [                                                                                                                    ; call each of one of the capitalist
      set Current-Income           Current-Income + div                                                                                  ; increase the income of the capitalist
      set Deposit-Bank             Deposit-Bank   + div                                                                                  ; update the bank account of the capitalist
      ]
    set Equity-Total               Equity-Total   - Profit
    set Profit                     0
    ]
  ask Wholesale-banks [                                                                                                                  ; the wholesalers bank dividend
    set Profit                     max list 0 (Interest-Receive - (Intrabank-loan * Intrabank-rate)) *  Dividend-payout-ratio            ; check if there was profit this period                                                       ;
    let div                        Profit  / (Number-of-consumptions-firms + number-of-capital-goods-firms)                              ; calculate dividend as part of the total profit divided by capitalists
    ask Capitalists [                                                                                                                    ; call each of one of the capitalist
      set Current-income           current-income + div                                                                                  ; increase the income of the capitalist
      set deposit-bank             deposit-bank   + div                                                                                  ; update the bank account of the capitalist
      ]
    set Equity-Total               Equity-Total - Profit
  ]
end



to SubF-Call-Matrix-formation                                                                                              ; subfunction that manage the R rotines
  ask C-Firms [
    set Matrix-C-Firms-Bankruptcy matrix:set-and-report Matrix-C-Firms-Bankruptcy 1 Mj Leverage                            ; update the matrix with information about C-Firms
    ifelse (Retail-Bankruptcy < T-Ban) and (Wholesale-Bankruptcy < T-Ban)                                                  ; update a position with 0 if the firm is alive, and 1 if the firm went bankruptcy
    [ set Matrix-C-Firms-Bankruptcy matrix:set-and-report Matrix-C-Firms-Bankruptcy 0 Mj 0 ]
    [ set Matrix-C-Firms-Bankruptcy matrix:set-and-report Matrix-C-Firms-Bankruptcy 0 Mj 1 ]
    set Mj Mj + 1 if Mj > (Window - 1) [ set Mj 0]                                                                         ; move the matrix to the next position
    ]

  ask K-Firms [
    set Matrix-K-Firms-Bankruptcy matrix:set-and-report Matrix-K-Firms-Bankruptcy 1 Mjk leverage                           ; update the matrix with information abou the K-firms
    ifelse (Retail-Bankruptcy < T-Ban) and (Wholesale-Bankruptcy < T-Ban)                                                  ; update a position with 0 if the if alive, and 1 if the firm went bankruptcy
    [ set Matrix-K-Firms-Bankruptcy matrix:set-and-report Matrix-K-Firms-Bankruptcy 0 Mjk 0 ]
    [ set Matrix-K-Firms-Bankruptcy matrix:set-and-report Matrix-K-Firms-Bankruptcy 0 Mjk 1 ]
    set Mjk Mjk + 1 if Mjk > (Window - 1) [ set Mjk 0]                                                                     ; move the matrix the next position
    ]

  let A matrix:get-row Matrix-C-Firms-Bankruptcy 0                                                                         ; creat an object with the first row of the matrix
  let B matrix:get-row Matrix-C-Firms-Bankruptcy 1                                                                         ; creat an object with the seccond row of the matrix

  r:put   "a" A                                                                                                            ; call the R and put there the object A
  r:put   "b" B                                                                                                            ; call the R and put there the object B
  r:eval  "c <- glm(a ~ b, family = binomial)"                                                                             ; call R and ask to solve the logistic function with the inputs A and B

  let C  r:get "c$coefficients"                                                                                            ; call R and the coefficients into object C
  set Beta_1-C-Firms  first C                                                                                              ; save the beta-1 for C-firms
  if  Beta_1-C-Firms < (- 25)      [set Beta_1-C-Firms (- 25)]                                                             ; assure value into a meaninful range, because the first period there are not enough information
  set Beta_2-C-Firms  last  C                                                                                              ; sabe the beta-2 for C-firms
  if  Beta_2-C-Firms > 0  = false  [set Beta_2-C-Firms      0]                                                             ; assure value into a meaninful range, because the first period there are not enough information
  if  Beta_2-C-Firms > 25          [set Beta_2-C-Firms     25]                                                             ; assure value into a meaninful range, because the first period there are not enough information

  set A matrix:get-row Matrix-K-Firms-Bankruptcy 0                                                                         ; creat an object with the first row of the matrix
  set B matrix:get-row Matrix-K-Firms-Bankruptcy 1                                                                         ; creat an object with the seccond row of the matrix

  r:put   "a" A                                                                                                            ; call the R and put there the object A
  r:put   "b" B                                                                                                            ; call the R and put there the object B
  r:eval  "c <- glm(a ~ b, family = binomial)"                                                                             ; call R and ask to solve the logistic function with the inputs A and B

  set C  r:get "c$coefficients"                                                                                            ; the same as for the C-firms, look above for details
  set Beta_1-K-Firms  first C
  if  Beta_1-K-Firms < (- 25)     [set Beta_1-K-Firms  (- 25)]
  set Beta_2-K-Firms  last  C
  if (Beta_2-K-Firms > 0) = false [set Beta_2-K-Firms       0]
  if  Beta_2-K-firms > 25         [set Beta_2-K-Firms      25]

end


To SubF-Call-Wholesales                                  ; subfunction used for firms to ask credit at the wholesalers banks
    ask Firms [
      SubF-Update-Installment-W                          ; call subfunction to update installment
      SubF-Update-Cash-Need                              ; call subfunction to update possible need for cash
      SubF-Update-Assets                                 ; call subfunction to update the assets
      SubF-Update-Leverage-W                             ; call subfunction to update leverage
    ]
    ask K-firms [
      SubF-Ask-for-K-Credit-W                            ; check with a K-firm if it needs credit
    ]
    ask C-Firms [
      SubF-Ask-for-C-Credit-W                            ; check with a C-firm if it needs credit
    ]

end


to SubF-Update-Installment-W                                                                              ; subfunction to update the installment wiht the wholesalers bank
  set  Interest-Pay                     0                                                                 ; reset parameter
  set  Installment-Pay                  0                                                                 ; reset parameter
  let  Interest                         0                                                                 ; reset parameter
  let  Installment                      0                                                                 ; reset parameter
  if Current-Wholesale-Loan > 0 [                                                                         ; check if this firm already has a loan with this bank
    ifelse Bank-deposit     >= ((Installment-on-debt-wholesales                 + Interest-Wholesale ) * Current-Wholesale-Loan) [                     ; check if this firm has enough money to pay their debits
      set Interest-Pay                Interest-Wholesale             * Current-Wholesale-Loan             ; update the parameter
      set Bank-Deposit                Bank-Deposit                   - interest-pay                       ; updating bank account
      set Interest                    interest-pay                                                        ; update parameter
      set Installment-Pay             Installment-on-debt-Wholesales * Current-Wholesale-Loan                ; update parameter
      set Bank-Deposit                Bank-Deposit                   - Installment-Pay                    ; update bank account
      set Current-Wholesale-Loan      Current-Wholesale-Loan         - Installment-Pay                    ; reduce the debit after the payment
      set Installment                 Installment-pay                                                     ;
      set Wholesale-Bankruptcy        0                                                                   ; means the firm did not went bankruptcy
      ] [
    set Wholesale-Bankruptcy          Wholesale-Bankruptcy  +  1                                          ; if the firm did not had money she will be closer and closer to bankruptcy
      ]
    ]
  ask Wholesale-Banks [                                                                                   ; update the wholesalers bank properties
    set Equity-Total                  Equity-Total     + Installment
    set Interest-Receive              Interest-Receive + Interest
    ]
end


to SubF-Update-Leverage-W                                                                                 ; subfuntion for update the leverage of firms with the wholesalers bank
  if Need-Cash > 0                    [ set Leverage                      (Current-Wholesale-Loan + Need-Cash) / (Assets + Current-Wholesale-Loan + Need-Cash)] ;
  if (Current-Wholesale-Loan = 0)     [ set Leverage 0 ]
end

to SubF-Ask-for-K-Credit-W                                                                                ; subfunction used by a K-firms to ask credit with a wholesaler bank
  let Availible-Credit              Bank-loss-parameter * (sum [Equity-Total] of Wholesale-Banks * B-Lev )  ; check money availible for new loans
  if (Need-Cash > 0)  and (Availible-Credit > Need-Cash) [                                                ; check if this firm had luck and receive a new loan
    set Current-Wholesale-Loan       Current-Wholesale-Loan  +  Need-Cash                                 ; creating a new loan
    set Bank-Deposit                 Bank-Deposit         +  Need-Cash                                    ; update bank account wiht new credit reeceived
    ifelse Update-Interest?          [ SubF-Update-Interest-Wholesale-K ] [ set Interest-Wholesale              free-interest-rate] ; call subfunction to update the interest rate with the wholesaler bank
    ]
end

to SubF-Ask-for-C-Credit-W                                                                                ; subfunction used by a C-firm to ask for credit with a wholsaler bank
  let Availible-Credit              Bank-loss-parameter * (sum [Equity-Total] of Wholesale-Banks * B-Lev )  ; check money availible for new loans
  if (Need-Cash > 0)  and (Availible-Credit > Need-Cash) [                                                ; check if this firm had luck and receive a new loan
    set Current-Wholesale-Loan       Current-Wholesale-Loan  +  Need-Cash                                 ; creating a loan
    set Bank-Deposit                 Bank-Deposit            +  Need-Cash                                 ; update bank account wiht new credit reeceived
    ifelse Update-Interest?          [ SubF-Update-Interest-Wholesale-C ] [ set Interest-Wholesale              free-interest-rate] ; call subfunction to update the interest rate with the wholesaler bank
    ]
end

to SubF-Update-Interest-Wholesale-C                                                                       ; this function update the interest rate of a firm with the wholesaler bank. Similar to the subfunction for the retail bank, check above for details
  let prob-die                       (1 / (1 + e ^ ( - Beta_1-C-Firms - (Beta_2-C-Firms * leverage))))
  set T-expected                     (1 / prob-die)
  let Numerator                      (1 + (free-interest-rate /                         installment-on-debt-wholesales))
  let Denominator                    ((1 - (1 - installment-on-debt-wholesales) ^ (T-expected + 1)) / installment-on-debt-wholesales)
  let new-int                        Bank-gross-mark-up    * ((numerator / denominator)  - installment-on-debt-wholesales)
  set Interest-Wholesale             (Interest-Wholesale   * (Current-Wholesale-Loan - Need-Cash) + new-int * Need-Cash) / Current-Wholesale-Loan
end

to SubF-Update-Interest-Wholesale-K                                                                       ; this function update the interest rate of a firm with the wholesaler bank. Similar to the subfunction for the retail bank, check above for details
  let prob-die                       (1 / (1 + e ^ ( - Beta_1-K-Firms - (Beta_2-K-Firms * leverage))))
  set T-expected                     (1 / prob-die)
  let Numerator                      (1 + (free-interest-rate /                         installment-on-debt-Wholesales))
  let Denominator                    ((1 - (1 - installment-on-debt-wholesales) ^ (T-expected + 1)) / installment-on-debt-Wholesales)
  let new-int                        Bank-gross-mark-up    * ((numerator / denominator) -  installment-on-debt-Wholesales)
  set Interest-Wholesale             (Interest-Wholesale   * (Current-Wholesale-Loan - Need-Cash) + new-int * Need-Cash) / Current-Wholesale-Loan
end




to SubF-Call-Intrabank                                                                                   ; this subfunction has the process for the intrabank structure
  let lev                            sum [Current-Wholesale-Loan] of Firms                               ; auxiliar variable with the value of the total wholesaler loans
  let New-Intra-Loan                 0                                                                   ; local auxiliar variable
  let Availible-Loan                 0                                                                   ; local auxiliar variable
  let Installment                    0                                                                   ; local auxiliar variable
  let Interest                       0                                                                   ; local auxiliar variable
  let Need-loan                       max list 0 ( Initial-equity-of-the-bank - (abs sum [Intrabank-loan] of Wholesale-Banks) + lev)  ; local variable with the value of the new loans needed for the wholesaler bank
  if Need-Loan > 0 [                                                                                     ; if the wholesaler bank need more money...
    ask Retail-Banks [                                                                                   ; ... then ask the retail bank if there is any availible
      set Availible-Loan             min list Need-Loan (Bank-loss-parameter * Equity-Total * (number-of-consumptions-firms + number-of-capital-goods-firms))       ;  availible credit for this period
      set Intrabank-Loan             Intrabank-Loan - Availible-Loan                                     ; update the intrabank credit value
      set Equity-Total               Equity-Total   - Availible-Loan                                     ; update the equity of the retail bank
      ]
    ]
  ask Wholesale-Banks [                                                                                  ; update the parameter of the wholesalers bank after new loans
    if (Intrabank-Loan > 0) and (Equity-Total > 0) [                                                     ; proceed  with the payment of debits of the Wholesaler bank with the retail bank
      set Installment                Intrabank-Loan * (Installment-on-debt)                              ; update the installment payed
      set Interest                   Intrabank-Loan * Intrabank-Rate                                     ; update the interest payed
      set Equity-Total               Equity-Total   - Installment - Interest                             ; update equity of the wholesaler bank
      set Intrabank-Loan             Intrabank-Loan - (Intrabank-Loan * Installment-on-debt)             ; update parameter of intrabank loan
      ]
    set Equity-Total                 Equity-total   + Availible-Loan                                     ; this is called even when there was not previous intrabank credit
    set Intrabank-Loan               Intrabank-Loan + Availible-Loan                                     ; this is called even when there was not previous intrabank credit
    if Equity-total < 0 [ set Equity-total Initial-equity-of-the-bank]                                   ; bank bankruptcy, recapitalized
    ]
  ask Retail-Banks [                                                                                     ; update the retail bank properties using the local auxiliar variables
    set Equity-Total                 Equity-Total   + Installment + Interest
    set Intrabank-Loan               Intrabank-Loan + Installment
    ]

end




;*************************************************************************************Function-Call-Central-Bank*************************************************************************************************




to Function-Call-Central-Bank
  ;Taylor-rule-parameter-for-product
  ;Taylor-rule-parameter-for-inflation
  ;Desire-inflation-for-the-monetary-authority
  ;Natural-interest-rate
  let pi_t                          ((ln Goods-Price-Level - ln Goods-Price-Level_t-1) + (ln Capital-Price-Level - ln Capital-Price-Level_t-1))
  let P_gdp                         Productivity-of-capital * sum [Capital] of C-Firms
  let sensity_inflation             Taylor-rule-parameter-for-inflation * (pi_t - Desire-inflation-for-the-monetary-authority)
  let sensity_product               Taylor-rule-parameter-for-product   * (ln GDP - ln P_gdp)
  set Free-Interest-Rate            max list 0.01 ( (pi_t + natural-interest-rate + sensity_inflation + sensity_product) * (Slow_taylor) + (1 - slow_taylor) * Free-Interest-Rate )

end







@#$#@#$#@
GRAPHICS-WINDOW
5
10
523
223
-1
-1
20.4
1
10
1
1
1
0
1
1
1
0
24
0
9
1
1
1
ticks
30.0

SLIDER
786
10
1006
43
Number-of-periods
Number-of-periods
0
10000
3000.0
100
1
Quarters
HORIZONTAL

SLIDER
786
42
1006
75
Number-of-workers
Number-of-workers
0
10000
2700.0
100
1
Workers
HORIZONTAL

SLIDER
786
74
1006
107
Number-of-consumptions-firms
Number-of-consumptions-firms
0
400
200.0
1
1
NIL
HORIZONTAL

SLIDER
786
106
1006
139
Number-of-capital-goods-firms
Number-of-capital-goods-firms
0
100
50.0
1
1
NIL
HORIZONTAL

SLIDER
786
139
1006
172
Number-of-firms-visited-by-unemployed-workers
Number-of-firms-visited-by-unemployed-workers
0
10
9.0
1
1
NIL
HORIZONTAL

SLIDER
786
171
1006
204
Number-of-consumptions-firms-visited-by-a-consumer
Number-of-consumptions-firms-visited-by-a-consumer
0
10
2.0
1
1
NIL
HORIZONTAL

SLIDER
786
204
1006
237
Number-of-capital-goods-firms-visited-by-a-C-Firm
Number-of-capital-goods-firms-visited-by-a-C-Firm
0
10
2.0
1
1
NIL
HORIZONTAL

SLIDER
786
237
1006
270
Memory-parameter-human-wealth
Memory-parameter-human-wealth
0
1
0.96
0.01
1
NIL
HORIZONTAL

SLIDER
786
269
1006
302
Dividend-payout-ratio
Dividend-payout-ratio
0
1
0.2
.01
1
NIL
HORIZONTAL

SLIDER
786
302
1006
335
Fraction-of-wealth-devoted-to-consumption
Fraction-of-wealth-devoted-to-consumption
0
1
0.05
.01
1
NIL
HORIZONTAL

SLIDER
786
335
1006
368
Initial-risk-free-interest-rate
Initial-risk-free-interest-rate
0
0.1
0.01
.005
1
NIL
HORIZONTAL

SLIDER
786
369
1006
402
Quantity-adjust-parameter
Quantity-adjust-parameter
0
1
0.9
0.01
1
NIL
HORIZONTAL

SLIDER
788
403
1009
436
Price-adjust-parameter
Price-adjust-parameter
0
0.2
0.1
.01
1
NIL
HORIZONTAL

SLIDER
1140
10
1364
43
Bank-gross-mark-up
Bank-gross-mark-up
1
2
1.2
0.01
1
NIL
HORIZONTAL

SLIDER
1731
79
1903
112
!
!
0
100
50.0
1
1
NIL
HORIZONTAL

SLIDER
1140
42
1364
75
Productivity-of-labor
Productivity-of-labor
0
1
0.5
.01
1
NIL
HORIZONTAL

SLIDER
1140
74
1364
107
Productivity-of-capital
Productivity-of-capital
0
1
0.33
.01
1
NIL
HORIZONTAL

SLIDER
1140
106
1364
139
Probability-of-investing
Probability-of-investing
0
1
0.25
.01
1
NIL
HORIZONTAL

SLIDER
1140
139
1364
172
Bank-loss-parameter
Bank-loss-parameter
0
.01
0.002
0.001
1
NIL
HORIZONTAL

SLIDER
1140
172
1364
205
Installment-on-debt
Installment-on-debt
0
.2
0.05
0.01
1
NIL
HORIZONTAL

SLIDER
1140
205
1364
238
Depreciation-of-capital
Depreciation-of-capital
0
.1
0.02
0.005
1
NIL
HORIZONTAL

SLIDER
1140
237
1364
270
Memory-parameter-investment
Memory-parameter-investment
0
1
0.5
.1
1
NIL
HORIZONTAL

SLIDER
1140
270
1364
303
Desire-capacity-utilization-rate
Desire-capacity-utilization-rate
.5
1
0.85
.01
1
NIL
HORIZONTAL

SLIDER
1140
302
1364
335
Wage
Wage
.25
2
1.0
.01
1
NIL
HORIZONTAL

SLIDER
1140
335
1364
368
Initial-liquidity-of-Firms
Initial-liquidity-of-Firms
5
30
10.0
1
1
NIL
HORIZONTAL

SLIDER
1140
367
1364
400
Initial-capital
Initial-capital
5
30
5.0
1
1
NIL
HORIZONTAL

SLIDER
1140
400
1364
433
Initial-production-C-firms
Initial-production-C-firms
1
20
5.0
1
1
NIL
HORIZONTAL

SLIDER
1466
10
1690
43
Initial-production-K-firms
Initial-production-K-firms
1
10
2.0
1
1
NIL
HORIZONTAL

SLIDER
1466
41
1690
74
Initial-equity-of-the-bank
Initial-equity-of-the-bank
0
10000
3000.0
500
1
NIL
HORIZONTAL

SLIDER
1466
74
1690
107
Initial-households-personal-assets
Initial-households-personal-assets
1
10
2.0
1
1
NIL
HORIZONTAL

SLIDER
1466
106
1690
139
Natural-interest-rate
Natural-interest-rate
0
.02
0.01
.005
1
NIL
HORIZONTAL

SLIDER
1466
139
1697
172
Desire-inflation-for-the-monetary-authority
Desire-inflation-for-the-monetary-authority
0
1
1.0
.01
1
NIL
HORIZONTAL

SLIDER
1466
172
1690
205
Taylor-rule-parameter-for-inflation
Taylor-rule-parameter-for-inflation
0
1
1.0
0.05
1
NIL
HORIZONTAL

SLIDER
1466
204
1690
237
Taylor-rule-parameter-for-product
Taylor-rule-parameter-for-product
0
1
1.0
0.05
1
NIL
HORIZONTAL

BUTTON
20
254
84
287
Setup
Setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
19
379
127
412
Workers?
Workers?
0
1
-1000

SWITCH
19
346
127
379
Capitalist?
Capitalist?
0
1
-1000

SWITCH
20
445
127
478
C-Firm?
C-Firm?
0
1
-1000

SWITCH
20
412
127
445
K-Firm?
K-Firm?
0
1
-1000

SWITCH
20
478
127
511
Bank?
Bank?
0
1
-1000

TEXTBOX
29
324
122
342
Setup Conditions:
11
0.0
1

BUTTON
82
254
145
287
Go
Go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
145
254
208
287
Go
Go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
1512
470
1887
620
Unemployment Rate
NIL
NIL
0.0
1.0
1.0
0.0
true
false
"" ""
PENS
"Unemployment" 1.0 0 -16777216 true "" "if ticks > 100 [plot Unemployment]"
"Income (x1000)" 1.0 0 -10899396 true "" ";if ticks > 10 [plot (sum [Current-Income] of households) / 3000]"

MONITOR
1046
10
1103
55
Unemp
Unemployment
2
1
11

PLOT
220
683
420
833
Stock K-Firms
NIL
NIL
0.0
10.0
1000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [stock] of K-Firms]"

PLOT
419
683
619
833
Capital C-sector
NIL
NIL
0.0
10.0
10000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 300 [plot sum [Capital] of C-Firms]"

PLOT
1137
622
1511
772
Investment
NIL
NIL
0.0
10.0
1000.0
10.0
true
false
"" ""
PENS
"Investment" 1.0 0 -16777216 true "" "if ticks > 100 [plot Total-Investment]"
"K-Firms Sales" 1.0 0 -955883 true "" "if ticks > 100 [plot sum [salles] of K-Firms]"

PLOT
620
685
820
835
Production K-Firms
NIL
NIL
0.0
10.0
1000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [Production] of K-Firms]"

PLOT
22
682
222
832
Workers C-Sector
NIL
NIL
0.0
10.0
10000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [Number-Employee] of C-Firms]"

PLOT
22
533
222
683
Workers K-Sector
NIL
NIL
0.0
10.0
1000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [Number-employee] of K-Firms]"

MONITOR
371
850
442
895
K-Workers
sum [number-employee] of K-Firms
2
1
11

MONITOR
301
850
373
895
C-Workers
sum [number-employee] of C-firms
2
1
11

MONITOR
444
894
501
939
Dep
sum [capital] of C-firms * depreciation-of-capital * mean [Util-capacity] of c-firms
2
1
11

MONITOR
500
894
557
939
K-Prod
sum [production] of k-firms
2
1
11

PLOT
222
533
530
683
des x Investment
NIL
NIL
0.0
10.0
1000.0
10.0
true
true
"" ""
PENS
"des-invest" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [desire-investment] of C-firms]"
"invest" 1.0 0 -13791810 true "" "if ticks > 100 [plot total-investment]"

MONITOR
558
894
615
939
K-Stock
sum [stock] of k-firms
2
1
11

MONITOR
442
850
499
895
Inves
total-investment
2
1
11

MONITOR
617
893
674
938
K-sale
sum [salles] of K-firms
2
1
11

MONITOR
444
938
501
983
C-Inv
sum [Investment] of C-Firms
2
1
11

PLOT
1135
470
1510
620
GDP
NIL
NIL
0.0
10.0
10000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot GDP]"

MONITOR
560
849
618
894
C-Rev
sum [Revenue] of C-Firms
2
1
11

MONITOR
617
849
675
894
K-Rev
sum [revenue] of K-Firms
1
1
11

MONITOR
330
895
388
940
C-Prof
sum [profit] of C-Firms
1
1
11

MONITOR
388
893
446
938
K-Prof
sum [profit] of K-firms
1
1
11

MONITOR
739
1098
797
1143
C-Dep
sum [bank-deposit] of C-firms
0
1
11

MONITOR
825
1099
883
1144
K-Dep
sum [bank-deposit] of k-firms
0
1
11

MONITOR
781
1143
839
1188
F-dep
sum [bank-deposit] of firms
0
1
11

MONITOR
568
1094
707
1139
Liq-C-Sec
sum [profit] of C-Firms - sum [revenue] of K-Firms
0
1
11

PLOT
366
1054
566
1204
net C-Sector
NIL
NIL
0.0
10.0
0.0
-2000.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot sum [profit] of C-Firms - sum [revenue] of K-Firms]"
"pen-1" 1.0 0 -8053223 true "" ";if ticks > 100 [ plot 0]"

BUTTON
93
978
157
1012
Go
Go
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
32
985
97
1019
Setup
Setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
230
1010
294
1044
go
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
289
383
432
416
K-UpDate-Price?
K-UpDate-Price?
0
1
-1000

SWITCH
292
418
438
451
K-Qtt-Function?
K-Qtt-Function?
0
1
-1000

SWITCH
286
479
428
512
C-Qtt-Function?
C-Qtt-Function?
0
1
-1000

SWITCH
286
449
430
482
C-UpDate-Price?
C-UpDate-Price?
0
1
-1000

MONITOR
593
75
651
120
K-Price
Capital-Price-Level
2
1
11

MONITOR
659
78
717
123
C-Price
Goods-Price-Level
2
1
11

PLOT
533
536
891
686
Prices
NIL
NIL
0.0
10.0
10.0
0.0
true
true
"" ""
PENS
"C-Price" 1.0 0 -16777216 true "" "if ticks > 5 [plot Goods-Price-Level]"
"K-Price" 1.0 0 -5298144 true "" "if ticks > 5 [plot Capital-Price-Level]"

PLOT
706
986
906
1136
Total-Money
NIL
NIL
0.0
10.0
8900.0
9100.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot (sum [deposit-bank] of households + sum [bank-deposit] of firms)"

PLOT
1103
834
1444
984
Hous-Money
NIL
NIL
0.0
10.0
1.0
0.0
true
true
"" ""
PENS
"Workers" 1.0 0 -5298144 true "" "plot sum [deposit-bank] of workers"
"Capitalist" 1.0 0 -14070903 true "" "plot sum [deposit-bank] of capitalists"

PLOT
764
833
1105
983
Firm-Money
NIL
NIL
0.0
10.0
1.0
0.0
true
true
"" ""
PENS
"C-Firms" 1.0 0 -5298144 true "" "plot sum [bank-deposit] of c-firms"
"k-Firms" 1.0 0 -14070903 true "" "plot sum [bank-deposit] of K-firms"

SWITCH
429
448
613
481
c-Firm_Budget_rest01?
c-Firm_Budget_rest01?
0
1
-1000

SWITCH
429
481
615
514
C-Firm_Budget_rest02?
C-Firm_Budget_rest02?
0
1
-1000

PLOT
1513
622
1886
772
Consumption
NIL
NIL
0.0
10.0
10000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 100 [plot Consumption]"

SWITCH
126
478
285
511
Retail-Banks?
Retail-Banks?
0
1
-1000

SWITCH
125
445
287
478
WholeSales-Banks?
WholeSales-Banks?
1
1
-1000

PLOT
906
985
1106
1135
C-Loans
NIL
NIL
0.0
10.0
10000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Total-Loans-C-Firms"

PLOT
1104
984
1304
1134
K-Loans
NIL
NIL
0.0
10.0
10000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Total-Loans-K-Firms"

SWITCH
128
415
293
448
Update-Interest?
Update-Interest?
0
1
-1000

PLOT
1472
989
1672
1139
Total-Loans
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot total-loans"
"pen-1" 1.0 0 -5298144 true "" "plot sum [Equity-Bank] of Retail-Banks * B-Lev"

MONITOR
1592
923
1667
968
T-Loans
Total-loans
1
1
11

CHOOSER
66
911
205
956
B-Lev
B-Lev
1 5 10
2

CHOOSER
66
865
205
910
T-Ban
T-Ban
1 2 4
1

PLOT
1700
775
1900
925
Hist Prod C-Firm
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" "if ticks > 20 [histogram [production] of C-Firms]"

PLOT
1709
932
1909
1082
Hist K-Production
NIL
NIL
0.0
10.0
0.0
10.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" "if ticks > 20 [histogram [production] of K-Firms]"

SWITCH
129
348
289
381
C-Broke?
C-Broke?
0
1
-1000

SWITCH
128
379
290
412
K-Broke?
K-Broke?
0
1
-1000

PLOT
892
535
1138
685
C-Broke
NIL
NIL
0.0
10.0
100.0
1.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" "plot c-broke"

MONITOR
499
939
557
984
C-Broke
C-Broke
0
1
11

PLOT
893
685
1141
835
K-Broke
NIL
NIL
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 1 -16777216 true "" "plot K-Broke"

MONITOR
389
939
445
984
C-Loan mean
mean [Current-Retail-Loan] of C-firms
0
1
11

MONITOR
330
941
389
986
K-Loan mean
mean [Current-Retail-Loan] of K-Firms
0
1
11

MONITOR
502
850
560
895
C-Price
goods-price-level
1
1
11

SWITCH
289
349
431
382
Profit?
Profit?
0
1
-1000

SWITCH
429
416
615
449
Capital-Destruction?
Capital-Destruction?
1
1
-1000

SWITCH
430
383
618
416
Prof-K-Firms?
Prof-K-Firms?
1
1
-1000

PLOT
40
1065
200
1215
C-leverage
NIL
NIL
0.0
10.0
1.0
0.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot mean [leverage] of C-Firms"

PLOT
196
1063
356
1213
k-Leverage
NIL
NIL
0.0
10.0
1.0
0.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot mean [leverage] of K-Firms"

PLOT
1399
313
1559
463
credit-restriction
NIL
NIL
0.0
10.0
1.0
0.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot  ( 1 / exp(- ( - 5.5 + 12 * mean [leverage] of C-firms) ))"

PLOT
1555
313
1715
463
Credit-Restriction-K
NIL
NIL
0.0
10.0
1.0
0.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot  ( 1 / exp(- ( - 10.3 + 12 * mean [leverage] of K-firms) ))"

PLOT
1905
772
2105
922
Beta 1
NIL
NIL
0.0
10.0
1.0
0.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Beta_1-C-Firms"

PLOT
2102
770
2302
920
Beta 2
NIL
NIL
0.0
10.0
10.0
0.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Beta_2-C-Firms"

MONITOR
892
492
950
537
Beta-1
Beta_1-C-Firms
1
1
11

MONITOR
948
492
1006
537
Beta-2
Beta_2-C-Firms
1
1
11

MONITOR
1013
493
1078
538
Beta_1-K
Beta_1-K-Firms
1
1
11

MONITOR
1074
493
1139
538
Beta_2-K
Beta_2-K-Firms
1
1
11

MONITOR
824
468
882
513
C-Time
1 / (1 / (1 + e ^ ( - Beta_1-C-Firms - (Beta_2-C-Firms * mean [leverage] of C-Firms))))
2
1
11

PLOT
919
328
1119
478
C-Time-Life
NIL
NIL
0.0
10.0
1.0
0.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "if ticks > 1000 [ plot 1 / (1 / (1 + e ^ ( - Beta_1-C-Firms - (Beta_2-C-Firms * mean [leverage] of C-Firms)))) ]"

SWITCH
433
350
568
383
Random-Seed?
Random-Seed?
0
1
-1000

CHOOSER
1873
15
2012
60
Window
Window
600 800 1600 3200 6400 10000
3

SLIDER
1720
182
1948
215
Installment-on-debt-Wholesales
Installment-on-debt-Wholesales
0
0.5
0.025
0.01
1
NIL
HORIZONTAL

PLOT
1893
465
2093
615
Intrabank-Loans
NIL
NIL
0.0
10.0
1000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot sum [Intrabank-Loan] of Wholesale-Banks"

PLOT
1896
618
2096
768
Equity Wholesale
NIL
NIL
0.0
10.0
10000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot sum [Equity-total] of Wholesale-Banks"

PLOT
1899
308
2099
458
Wholesale-Loans
NIL
NIL
0.0
10.0
1000.0
10.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot sum [current-wholesale-loan] of firms"

SWITCH
569
349
785
382
Central-Bank?
Central-Bank?
0
1
-1000

PLOT
1911
933
2111
1083
Free interest rate
NIL
NIL
0.0
0.0
10.0
0.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot Free-Interest-rate"

MONITOR
1912
1082
1970
1127
FIR
free-interest-rate
4
1
11

SLIDER
1469
236
1695
269
Slow_taylor
Slow_taylor
0
1
0.2
0.01
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

(a general understanding of what the model is trying to show or explain)

## HOW IT WORKS

(what rules the agents use to create the overall behavior of the model)

## HOW TO USE IT

(how to use the model, including a description of each of the items in the Interface tab)

## THINGS TO NOTICE

(suggested things for the user to notice while running the model)

## THINGS TO TRY

(suggested things for the user to try to do (move sliders, switches, etc.) with the model)

## EXTENDING THE MODEL

(suggested things to add or change in the Code tab to make the model more complicated, detailed, accurate, etc.)

## NETLOGO FEATURES

(interesting or unusual features of NetLogo that the model uses, particularly in the Code tab; or where workarounds were needed for missing features)

## RELATED MODELS

(models in the NetLogo Models Library and elsewhere which are of related interest)

## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.1
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="hp_filters_test" repetitions="4" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 3000</exitCondition>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>sum [capital] of C-Firms</metric>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-inflation">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="1"/>
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="4"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-product">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Logistic-Regression-r-Extension" repetitions="1" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 3000</exitCondition>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>sum [capital] of C-Firms</metric>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="1"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-inflation">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-product">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="200"/>
      <value value="400"/>
      <value value="600"/>
      <value value="800"/>
      <value value="1600"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Logistic-Regression-r-Extension_3" repetitions="10" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 2500</exitCondition>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>sum [capital] of C-Firms</metric>
    <metric>sum [production] of K-Firms</metric>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-inflation">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-product">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="experiment" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 2500</exitCondition>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>sum [capital] of C-Firms</metric>
    <metric>sum [production] of K-Firms</metric>
    <metric>sum [Current-Retail-Loan] of firms</metric>
    <metric>sum [current-wholesale-loan] of firms</metric>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-product">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="1"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt-Wholesales">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-inflation">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Wholesales-2" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 2500</exitCondition>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>sum [capital] of C-Firms</metric>
    <metric>sum [production] of K-Firms</metric>
    <metric>sum [Current-Retail-Loan] of firms</metric>
    <metric>sum [current-wholesale-loan] of firms</metric>
    <metric>sum [Intrabank-Loan] of Wholesale-Banks</metric>
    <metric>Capital-Price-Level</metric>
    <metric>Goods-Price-Level</metric>
    <metric>sum [Current-Retail-Loan] of C-Firms</metric>
    <metric>sum [Current-Retail-Loan] of K-Firms</metric>
    <metric>sum [Current-Wholesale-Loan] of C-Firms</metric>
    <metric>sum [Current-Wholesale-Loan] of K-Firms</metric>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-product">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="1"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt-Wholesales">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Taylor-rule-parameter-for-inflation">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Central-Bank" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 3000</exitCondition>
    <metric>Free-Interest-Rate</metric>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>Goods-Price-Level</metric>
    <metric>Goods-Price-Level_t-1</metric>
    <metric>Capital-Price-Level</metric>
    <metric>Capital-Price-Level_t-1</metric>
    <metric>Total-Loans</metric>
    <metric>Total-Loans-C-Firms</metric>
    <metric>Total-Loans-K-Firms</metric>
    <metric>C-Firms</metric>
    <metric>K-Firms</metric>
    <metric>Workers</metric>
    <metric>Capitalists</metric>
    <metric>Retail-Banks</metric>
    <metric>Wholesale-Banks</metric>
    <metric>C-Broke</metric>
    <metric>K-Broke</metric>
    <metric>Beta_1-C-Firms</metric>
    <metric>Beta_2-C-Firms</metric>
    <metric>Beta_1-K-Firms</metric>
    <metric>Beta_2-K-Firms</metric>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="1"/>
      <value value="4"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-product" first="0" step="0.25" last="1"/>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-inflation" first="0" step="0.25" last="1"/>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Slow_taylor" first="0" step="0.25" last="1"/>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Central-Bank?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="1"/>
      <value value="5"/>
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
      <value value="0.01"/>
      <value value="0.1"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt-Wholesales">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Central-Bank-B" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 3000</exitCondition>
    <metric>Free-Interest-Rate</metric>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>Goods-Price-Level</metric>
    <metric>Goods-Price-Level_t-1</metric>
    <metric>Capital-Price-Level</metric>
    <metric>Capital-Price-Level_t-1</metric>
    <metric>Total-Loans</metric>
    <metric>Total-Loans-C-Firms</metric>
    <metric>Total-Loans-K-Firms</metric>
    <metric>C-Broke</metric>
    <metric>K-Broke</metric>
    <metric>Beta_1-C-Firms</metric>
    <metric>Beta_2-C-Firms</metric>
    <metric>Beta_1-K-Firms</metric>
    <metric>Beta_2-K-Firms</metric>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-product" first="0" step="0.25" last="1"/>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-inflation" first="0" step="0.25" last="1"/>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Slow_taylor">
      <value value="0"/>
      <value value="0.33"/>
      <value value="0.66"/>
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Central-Bank?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0"/>
      <value value="0.01"/>
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt-Wholesales">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
  </experiment>
  <experiment name="Central-Bank-C" repetitions="5" runMetricsEveryStep="true">
    <setup>setup</setup>
    <go>go</go>
    <exitCondition>ticks &gt; 2500</exitCondition>
    <metric>Free-Interest-Rate</metric>
    <metric>GDP</metric>
    <metric>Unemployment</metric>
    <metric>Consumption</metric>
    <metric>Total-Investment</metric>
    <metric>Goods-Price-Level</metric>
    <metric>Goods-Price-Level_t-1</metric>
    <metric>Capital-Price-Level</metric>
    <metric>Capital-Price-Level_t-1</metric>
    <metric>Total-Loans</metric>
    <metric>Total-Loans-C-Firms</metric>
    <metric>Total-Loans-K-Firms</metric>
    <metric>C-Broke</metric>
    <metric>K-Broke</metric>
    <metric>Beta_1-C-Firms</metric>
    <metric>Beta_2-C-Firms</metric>
    <metric>Beta_1-K-Firms</metric>
    <metric>Beta_2-K-Firms</metric>
    <enumeratedValueSet variable="Workers?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Window">
      <value value="3200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-capital">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms-visited-by-a-C-Firm">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capitalist?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Dividend-payout-ratio">
      <value value="0.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="T-Ban">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Capital-Destruction?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms">
      <value value="200"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-households-personal-assets">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Price-adjust-parameter">
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-gross-mark-up">
      <value value="1.2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Fraction-of-wealth-devoted-to-consumption">
      <value value="0.05"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Wage">
      <value value="1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-capital">
      <value value="0.33"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank-loss-parameter">
      <value value="0.002"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-firms-visited-by-unemployed-workers">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-periods">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt">
      <value value="0.05"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-product" first="0" step="0.33" last="1"/>
    <enumeratedValueSet variable="K-UpDate-Price?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Quantity-adjust-parameter">
      <value value="0.9"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Firm?">
      <value value="true"/>
    </enumeratedValueSet>
    <steppedValueSet variable="Taylor-rule-parameter-for-inflation" first="0" step="0.33" last="1"/>
    <enumeratedValueSet variable="Desire-capacity-utilization-rate">
      <value value="0.85"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Slow_taylor">
      <value value="0"/>
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-capital-goods-firms">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Probability-of-investing">
      <value value="0.25"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Central-Bank?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-equity-of-the-bank">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Productivity-of-labor">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="c-Firm_Budget_rest01?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Natural-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="B-Lev">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-workers">
      <value value="3000"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Desire-inflation-for-the-monetary-authority">
      <value value="0.01"/>
      <value value="0.1"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Random-Seed?">
      <value value="false"/>
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Broke?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="!">
      <value value="50"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-risk-free-interest-rate">
      <value value="0.01"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-liquidity-of-Firms">
      <value value="10"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-human-wealth">
      <value value="0.96"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Retail-Banks?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Update-Interest?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Number-of-consumptions-firms-visited-by-a-consumer">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Depreciation-of-capital">
      <value value="0.02"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Memory-parameter-investment">
      <value value="0.5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="K-Qtt-Function?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Profit?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-K-firms">
      <value value="2"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="C-Firm_Budget_rest02?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Bank?">
      <value value="true"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="WholeSales-Banks?">
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Installment-on-debt-Wholesales">
      <value value="0.025"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Initial-production-C-firms">
      <value value="5"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="Prof-K-Firms?">
      <value value="true"/>
    </enumeratedValueSet>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
