Globals [
  two-navies         ;; a way to count all ships on both sides of the battle
  ran-ship           ;; picks a random ship to show in the monitor
  ran-enemyship      ;; picks a random enemy ship to show in the monitor
]

breed [ ships a-ship ]
breed [ enemyships a-enemyship ]
breed [ admirals a-admiral ]
breed [ smokeclouds a-smokecloud ]

turtles-own [
  morale            ;; this is the desire to fight of each soldier
  damage            ;; this is a simulation of bullets. It is not visualized to keep from distracting the model user
  target            ;; this variable is used to aim at enemy ships
]

patches-own [
  crowded-seas
]

; ======================== Setup ====================
to setup
  clear-all
  reset-ticks
  make-battlefield               ;; this puts the ships into rows and places the admirals at the back of the columns
  make-ran-ships
end

;; In the following proceedures, ships are given an initial xy cor based in columns,
;; given a initial color to denote their side of the battle,
;; and given an initial morale. This morale value is equal for all ships on each side if that switch is turned on.
;; These setup instructions are the same for the admirals
to make-battlefield
  set-default-shape turtles "boat top"
  set two-navies turtles with [ breed = "Ships" or breed = "enemyships" ]
  make-Ships
  make-admirals
  make-enemyships
  ask patches [
    set pcolor 89
    set crowded-seas patches in-radius 3
  ]
end

to make-Ships
  create-Ships Navy-Size [
    setxy (random -10 - 3) (random 16 + random -16)
    set size 3
    set heading 90
    set color blue
    ifelse Start_Morale_Equal? [ set morale Starting_Desire_to_Fight_Ships - random 30 ]
      [ set morale (random 50) + 10 ]
  ]
end

to make-admirals
  create-admirals Num-Admirals [
    setxy -15 (random 10 + random -10)
    set heading 90
    set size 4
    set color blue
    ifelse Start_Morale_Equal? [ set morale Starting_Desire_to_Fight_Ships - random 30 ]
      [ set morale (random 50) + 10 ]
  ]
end

to make-enemyships
  create-enemyships Enemy-Navy-Size [
    setxy (random 10 + 3) (random 16 + random -16) ;could use (world-height * who / militia_size) ??
    set size 3
    set heading 270
    set color red
    ifelse Start_Morale_Equal? [ set morale Starting_Desire_to_Fight_Ships - random 30 ]
      [ set morale (random 50) + 10 ]
  ]
end

to make-ran-ships
  set ran-ship random navy-size
  set ran-enemyship random enemy-navy-size
end


;======================== Go ========================
to go
  visualize-morale   ;; this changes the shading of each soldier to show the user the morale spread
  fight-ships    ;; this is the set of commands that tell ships how much damage to take from battle
  give-commands      ;; this set of commands is how the admirals "instruct" their fleet
  declare-winner     ;; this displays a message to a user when a side of the battle has won
  generate-smoke     ;; creates smoke from cannon fire on the battle field that ships avoid and increases confusion
  tick
end

; ====================== Code =======================

;; this scales the colors based on the morale of each soldier. This doesn't happen on the first tick,
;; and it rechecks every .25 seconds
;; darker shades = low morale
;; lighter shades = high morale
to visualize-morale
  every .25 [
    ifelse ticks <= 1 [stop] [
      ask enemyships [ set color scale-color red morale -50 150]
      ask Ships [ set color scale-color blue morale -50 150 ]
      ask admirals [ set color scale-color blue morale -50 150 ]
    ]
  ]
end

;; this is how admirals instruct their fleet. The instructions are seen as positive to morale.
;; admiral_skill tells how often orders go out.
;; admiral_influences effects how far the cone directs the message of positive morale, in the naval battle this goes out in 360 degrees
;; morale is capped at 100 because the color-scaling gets washed out
to give-commands
  float
  every (20 / admiral-skill) [
    ask admirals [
      ask Ships in-cone (Admiral-influence * 2.5 ) 360 [
        set morale (morale + (training_level / 2))
        if morale > 100 [ set morale 99 ]

      ]
    ]
  ]
end

;; this displays a message to the user when one of the armies has no ships left, admirals are not counted
to declare-winner
  if (count Ships <= 0) or (count enemyships <= 0) [
    ifelse count Ships < count enemyships [ user-message (word "the British have won this battle with " count ships " Colonist ship(s) surrendering") ]
    [ user-message (word "the Colonists have won this battle with " count enemyships " British ship(s) surrendering")]
  ]
end

to generate-smoke
  every 4 [
    create-smokeclouds random 2
    ask smokeclouds [
      setxy random-xcor random-ycor
      set shape "cloud"
      set color grey
      set size 5
      if count smokeclouds >= 10 [ die ]
    ]
  ]
end



;==================================== Soldier Rules =================================
to fight-ships
  float
  take-damage
  desert
end

to float
  every .5 [
    ask enemyships [
      ifelse any? other turtles in-cone 5 20 [ rt 90 fd 1 ] [ rt -20 fd 2 ]
      if any? other turtles in-cone .5 360 [ set damage damage + .5]
      if any? other enemyships in-cone 2 360 [set morale morale + 10]
    ]
    ask ships [
      ifelse any? other turtles in-cone 5 20 [ rt 90 fd 1 ] [ rt -20 fd 2 ]
      if any? other turtles in-cone .5 360 [ set damage damage + .5]
    ]
    ask admirals [
      ifelse any? other turtles in-cone 5 20 [ rt 90 fd 1 ] [ rt -20 fd 2 ]
      if any? other turtles in-cone .5 360 [ set damage damage + .5]
    ]
  ]
end

;; take-damage simulates ships getting wounded in battle.
to take-damage
  ;; prevents damage before battle is set up completely
  ;; continues every 2 seconds
  if ticks >= 4 [
    ; xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx collision damage ask patch-here if turtles-on = 2
    every 2 [
      ask enemyships [
        ;; this makes the damage random based on distance from enemy (bullets were highly innacurate)
        set damage (random 10 )
        ;; if damage is high, it effects the soldier's morale
        if damage >= 5 [set morale (morale - 10) ]
        ;; if damage is really high, the soldier may die, also this effects the morale of other ships nearby
        if damage >= 7 [
          let die-chance random 100
          if die-chance >= 70 [ die ]
          damage-other-enemy ]
        if morale >= 90 [morale-up-other-enemy ]
      ]
      ;; this is the same code as above but for ships
      ask Ships [
        set damage ( random 10)
        if damage >= 5 [set morale (morale - 10) ]
        if damage >= 7 [
          let die-chance random 100
          if die-chance >= 70 [ die ]
          damage-other-militia ]
        if morale >= 90 [ morale-up-other-militia ]
      ]
      ask admirals [
        set damage ( random 10)
        if damage >= 5 [set morale (morale - 10) ]
        if damage >= 7 [
          let die-chance random 100
          if die-chance >= 70 [ die ]
          damage-other-militia ]
        if morale >= 90 [ morale-up-other-militia ]
      ]
    ]
  ]
end

to damage-other-enemy
;; this lowers morale when you see your neighbor get damage
  ask enemyships in-cone (training_level / 2) 360  [ set morale morale - 1]
;; also we want the damage to be recognized by the other side,
;; so this next code increases moral when damage is delt
  ask Ships in-cone 10 360 [
    set morale (morale + .5)
    if morale > 100 [ set morale 99 ]
  ]
end

to morale-up-other-enemy
;; this allows ships to ride your momentum and get some morale from your success
  ask enemyships in-cone (training_level / 2) 360 [
    set morale (morale + (training_level / 4 ))
;; this next line makes sure morale never goes over 100
    if morale > 100 [ set morale 99 ]
  ]
end

to damage-other-militia
;; this code is the Milita version of damage-other-enemy
  ask Ships in-cone ( training_level / 2 ) 360 [ set morale morale - 1]
  ask enemyships in-cone 10 360 [
    set morale (morale + .5)
    if morale > 100 [ set morale 99 ]
  ]
end

to morale-up-other-militia
;; this code is the Milita version of morale-up-other-enemy
  ask Ships in-cone (training_level / 2 ) 360 [
    set morale (morale + (training_level / 4 ))
    if morale > 100 [ set morale 99 ]
  ]
end


;; desert causes ships with very low morale to leave the battlefield
to desert
  if ticks >= 10 [
    ask enemyships [
      if morale <= 1 [ die ]
    ]
    ask Ships [
      if morale <= 1 [ die ]
    ]
  ]
end
@#$#@#$#@
GRAPHICS-WINDOW
335
16
805
487
-1
-1
14.0
1
10
1
1
1
0
1
1
1
-16
16
-16
16
0
0
1
ticks
30.0

SLIDER
43
55
215
88
Navy-Size
Navy-Size
5
15
13.0
1
1
NIL
HORIZONTAL

SLIDER
39
229
213
262
Admiral-Influence
Admiral-Influence
1
5
5.0
1
1
NIL
HORIZONTAL

SLIDER
43
99
215
132
Enemy-Navy-Size
Enemy-Navy-Size
5
15
5.0
1
1
NIL
HORIZONTAL

SLIDER
40
142
215
175
Num-Admirals
Num-Admirals
1
5
5.0
1
1
NIL
HORIZONTAL

BUTTON
17
13
81
46
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
90
14
153
47
NIL
go
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

PLOT
72
506
422
656
Fleet_Morale
ticks
# Colonial Ships
0.0
100.0
0.0
20.0
false
true
"" ""
PENS
"strong morale" 1.0 0 -13345367 true "" "plot count ships with [morale >= 50]"
"weak morale" 1.0 0 -15390905 true "" "plot count ships with [morale < 30]"

MONITOR
222
666
310
711
Count Militia
count ships
17
1
11

SLIDER
43
314
215
347
Training_Level
Training_Level
1
5
4.0
1
1
NIL
HORIZONTAL

MONITOR
314
665
469
710
Colonist Navy Morale Avg
mean [morale] of ships
2
1
11

SLIDER
42
272
214
305
Admiral-Skill
Admiral-Skill
1
4
4.0
1
1
NIL
HORIZONTAL

SLIDER
13
410
238
443
Starting_Desire_to_Fight_Ships
Starting_Desire_to_Fight_Ships
0
100
59.0
1
1
NIL
HORIZONTAL

SWITCH
14
370
184
403
Start_Morale_Equal?
Start_Morale_Equal?
1
1
-1000

SLIDER
18
450
250
483
Starting_Desire_to_Fight_Enemy
Starting_Desire_to_Fight_Enemy
0
100
17.0
1
1
NIL
HORIZONTAL

MONITOR
389
717
472
762
Avg Damage
mean [ damage ] of ships
2
1
11

PLOT
433
507
741
657
British_Morale
# British Ships
NIL
0.0
10.0
0.0
20.0
true
false
"" ""
PENS
"default" 1.0 0 -8053223 true "" "plot count enemyships with [morale >= 50]"
"pen-1" 1.0 0 -1604481 true "" "plot count enemyships with [morale <= 30]"

MONITOR
484
667
596
712
British Avg Morale
mean [morale] of enemyships
2
1
11

MONITOR
823
27
993
72
Ship 1  Morale Monitor
[ morale ] of turtle ran-ship
2
1
11

MONITOR
826
88
973
133
Ship 2 Morale Monitor
[ morale ] of turtle ( ran-ship - 1)
17
1
11

MONITOR
1009
28
1149
73
Ship 1 Damage Monitor
[ damage ] of turtle ran-ship
2
1
11

MONITOR
1008
87
1148
132
Ship 2 Damage Monitor
[ damage ] of turtle ( ran-ship - 1)
2
1
11

MONITOR
826
144
995
189
British Ship 1 Morale Monitor
[ morale ] of turtle ran-enemyship
2
1
11

MONITOR
823
201
992
246
British Ship 2 Morale Monitor
[ morale ] of turtle ran-enemyship
2
1
11

MONITOR
1008
143
1185
188
British Ship 1 Damage Monitor
[ damage ] of turtle ran-enemyship
2
1
11

MONITOR
1009
201
1186
246
British Ship 2 Damage Monitor
[ damage ] of turtle (ran-ship - 1)
2
1
11

@#$#@#$#@
## WHAT IS IT?

This is a model of a naval battle from the time period of the American Revolutiono. It shows how individual ships acted based on the other ships around them and how they reacted to commands given by admirals aboard flagships. These battles would have been very confusing, and success would have had to have emerged from the actions of each soldier on each ship.

Admirals are lead ships within this model. However, unlike most portrayals of battles from this time period, the user is not given control over these commanders. This can give the user a new perspective on the effectiveness of commanders in these kinds of battles.

Actual fighting, with bullets or cannon fire, is not visualized in this model, though ships do take damage and cannon fire clouds are generated to provide obstacles for the ships. This was done to provide clarity to the user of the main features of the model.

## HOW IT WORKS

The rules in this model are followed by each soldier. The commanders also have a few rules.

Ships have a morale value. They increase this morale when:

1. they hear a command from their Admiral
2. when the morale of nearby friendly ships increases.

They decrease their morale when:

1. they take damage, both through random chance of getting hit, and from collisions with other ships.
2. when morale of their nearby ships decreases

Training Level affects the rate of these increases and decreases.

Ships also become deserters when their morale drops too low.

Commanders follow one rule:

1. At intervals based on Admiral_Skill, give commands to troops in a cone based on Admiral_Influence. These commands raise morale for soldiers who hear them.


## HOW TO USE IT

To experiment with this model, users can set up a number of battle conditions.

Navy-Size and British_Navy_Size -- increase or decrease the starting army sizes

Num_Admirals -- change the number of Commanders each army starts with

Admiral_Skill -- how often an Admiral gives a command

Admiral_Influence -- how far a command carries across to nearby ships



## THINGS TO NOTICE

After selecting go, you can watch the shading of the ships change as their morale increases or decreases. You can start to look for patterns arise at different moments in the battle.

Plots -- these display the number of ships of each navy bucketed by morale values (high & low)

Counts -- these display the number of ships remaining in each navy

Avg Morale -- these display the average morale across each navy at the given moment


## THINGS TO TRY


1. At first, run the model with a high number of ships on one navy and a low number in another army. Does this effect how morale moves across the ships?
2. Play with the influence and skill of the admirals. What happens if one navy has many more leaders? Are the leaders influential? Are there other factors which might seem more influential in raising morale?
3. Try turning starting_morale_equal? on. Instead of getting a random morale value at the start of the fight, the ships now inherit the morale value you give them. What are scenarios when this might be more realistic in going into a battle?
4. This model has a few noticable differences from the land battle version. How does movement of the ships effect the spread of morale? How does the smaller number of ships available in the model effect morale increases or decreases?


## EXTENDING THE MODEL

Could you add individual soldiers to each ship? How might their morale contribute to an overall ship-morale?

What other factors of a naval battle are missing from this model? How might you introduce goals into the simulations, where each ship might want to reach a certain point of land or eliminate a certain number of enemy ships? These factors could be added to influence morale of the ships/soldiers.


## CREDITS AND REFERENCES

(a reference to the model's URL on the web if it has one, as well as any other necessary credits, citations, and links)
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

boat top
true
0
Polygon -7500403 true true 150 1 137 18 123 46 110 87 102 150 106 208 114 258 120 270 180 270 183 258 193 209 198 150 191 87 178 46 163 17
Rectangle -7500403 true true 133 89 165 165
Polygon -7500403 true true 105 180 75 195 225 195 195 180 120 165
Polygon -16777216 true false 150 270
Polygon -7500403 true true 135 270 135 285 165 285 165 270 135 255
Polygon -7500403 true true 135 90 45 105 255 105 165 90 120 90
Polygon -7500403 true true 75 165
Polygon -7500403 true true 90 135 15 165 285 165 210 135 105 135

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

chess knight
false
0
Line -16777216 false 75 255 225 255
Polygon -7500403 true true 90 255 60 255 60 225 75 180 75 165 60 135 45 90 60 75 60 45 90 30 120 30 135 45 240 60 255 75 255 90 255 105 240 120 225 105 180 120 210 150 225 195 225 210 210 255
Polygon -16777216 false false 210 255 60 255 60 225 75 180 75 165 60 135 45 90 60 75 60 45 90 30 120 30 135 45 240 60 255 75 255 90 255 105 240 120 225 105 180 120 210 150 225 195 225 210
Line -16777216 false 255 90 240 90
Circle -16777216 true false 134 63 24
Line -16777216 false 103 34 108 45
Line -16777216 false 80 41 88 49
Line -16777216 false 61 53 70 58
Line -16777216 false 64 75 79 75
Line -16777216 false 53 100 67 98
Line -16777216 false 63 126 69 123
Line -16777216 false 71 148 77 145
Rectangle -7500403 true true 90 255 210 300
Rectangle -16777216 false false 90 255 210 300

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cloud
false
0
Circle -7500403 true true 13 118 94
Circle -7500403 true true 86 101 127
Circle -7500403 true true 51 51 108
Circle -7500403 true true 118 43 95
Circle -7500403 true true 158 68 134

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

sheep
false
15
Circle -1 true true 203 65 88
Circle -1 true true 70 65 162
Circle -1 true true 150 105 120
Polygon -7500403 true false 218 120 240 165 255 165 278 120
Circle -7500403 true false 214 72 67
Rectangle -1 true true 164 223 179 298
Polygon -1 true true 45 285 30 285 30 240 15 195 45 210
Circle -1 true true 3 83 150
Rectangle -1 true true 65 221 80 296
Polygon -1 true true 195 285 210 285 210 240 240 210 195 210
Polygon -7500403 true false 276 85 285 105 302 99 294 83
Polygon -7500403 true false 219 85 210 105 193 99 201 83

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

wolf
false
0
Polygon -16777216 true false 253 133 245 131 245 133
Polygon -7500403 true true 2 194 13 197 30 191 38 193 38 205 20 226 20 257 27 265 38 266 40 260 31 253 31 230 60 206 68 198 75 209 66 228 65 243 82 261 84 268 100 267 103 261 77 239 79 231 100 207 98 196 119 201 143 202 160 195 166 210 172 213 173 238 167 251 160 248 154 265 169 264 178 247 186 240 198 260 200 271 217 271 219 262 207 258 195 230 192 198 210 184 227 164 242 144 259 145 284 151 277 141 293 140 299 134 297 127 273 119 270 105
Polygon -7500403 true true -1 195 14 180 36 166 40 153 53 140 82 131 134 133 159 126 188 115 227 108 236 102 238 98 268 86 269 92 281 87 269 103 269 113

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270
@#$#@#$#@
NetLogo 6.0.4
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 90 180
Line -7500403 true 150 150 210 180
@#$#@#$#@
0
@#$#@#$#@
