extensions [csv]

links-own [
  weight         ;; Weight given to end1 activation by end2
  inlayer        ;; Layer index of end1
]

breed [bias-nodes bias-node]
breed [input-nodes input-node]
breed [output-nodes output-node]
breed [hidden-nodes hidden-node]

turtles-own [
  activation     ;; Determines the nodes output
  err            ;; Used by backpropagation to feed error backwards
  layer          ;; Layer of network node is contained in. Used for agentset manipulation.
  estvar         ;; Estimated variance of error signal
  learncoeff     ;; Individual learning coefficient
  dropped?       ;; Boolean true if node currently dropped
]

globals [
  epoch-error    ;; Measurement of how many training examples the network got wrong in the epoch
  input-size     ;; Size of inputs
  hiddensizes    ;; Vector of layer sizes, determines net topology
  output-size    ;; Size of outputs
  traindata      ;; Training data
  testdata       ;; Testing data
]

;;;
;;; LOAD FILES
;;;
to load-files
  file-close-all
  file-open "mnist_train.csv"
  set traindata (list)
  repeat 20000 [
    set traindata lput (csv:from-row file-read-line) traindata
  ]
  file-close
  set testdata csv:from-file "mnist_test.csv"
end

;;;
;;; SETUP PROCEDURES
;;;

;; Set patches, shapes, and files (invariant under node change)
to setup
  clear-all
  ask patches [ set pcolor gray ]
  set-default-shape bias-nodes "bias-node"
  set-default-shape input-nodes "circle"
  set-default-shape output-nodes "circle"
  set-default-shape hidden-nodes "output-node"
  set-default-shape links "small-arrow-shape"
  load-files
end

;; Set up nodes and links, initialize values, and propagate so
;; that activations make sense
to init-net
  clear-links
  clear-turtles
  clear-plot
  setup-nodes
  setup-links
  recolor
  set-learncoeffs
  propagate
  reset-ticks
end

;; Create, initialize, and position the nodes in the network
to setup-nodes
  set input-size 400
  set hiddensizes read-from-string hidden-sizes-string
  set output-size 10
  let l-index 0
  let index 0
  
  create-bias-nodes 1 [
     setxy nodex l-index nodey l-index index (input-size + 1)
     set activation 1
     set layer l-index
     set dropped? false
  ]
  
  set index 1
  
  repeat input-size [
    create-input-nodes 1 [
      setxy nodex l-index nodey l-index index (input-size + 1)
      set activation ((random 2) * 2) - 1
      set layer l-index
      set dropped? false
    ]
    set index index + 1
  ]
  
  set l-index 1
  set index 0
  
  foreach hiddensizes [
    
    create-bias-nodes 1 [
       setxy nodex l-index nodey l-index index (? + 1)
       set activation 1
       set layer l-index
       set dropped? false
    ]
     
    set index 1
     
    repeat ? [
     create-hidden-nodes 1 [
       setxy nodex l-index nodey l-index index (? + 1)
       set activation ((random 2) * 2) - 1
       set layer l-index
       set dropped? false
     ]
     set index index + 1
    ]
    
    set l-index l-index + 1
    set index 0
  ]
  
  repeat output-size [
    create-output-nodes 1 [
      setxy nodex l-index nodey l-index index output-size
      set activation ((random 2) * 2) - 1
      set layer l-index
      set dropped? false
    ]
    set index index + 1
  ]
  ask turtles [set size 0.5]
end

;; Create and initialize links between nodes in the network
to setup-links
  let l-index 0
  repeat (length hiddensizes) [
   connect-all (turtles with [layer = l-index]) (hidden-nodes with [layer = (l-index + 1)])
   set l-index l-index + 1
  ]
  connect-all (turtles with [layer = l-index]) (output-nodes with [layer = (l-index + 1)])
end

;; Completely connect nodes1 to nodes2 with links
to connect-all [nodes1 nodes2]
  let r 1 / (sqrt (count nodes1))
  ask nodes1 [
    create-links-to nodes2 [
      set weight random-float (2 * r) - r
      set inlayer [layer] of one-of nodes1
    ]
  ]
end

;; Adjust color of nodes and edges according to values
to recolor
  ask turtles [
    set color item (step activation) [black white]
  ]
  let l-index 0
  let maxw 0
  repeat (length hiddensizes) + 1 [
   set maxw max [abs weight] of links with [inlayer = l-index]
   ask links with [inlayer = l-index] [
     let wquotient (weight / maxw)
     let colorstr (wquotient * 127)
     let colorvec (list (colorstr + 127) (127 - (abs colorstr)) (127 - colorstr) 196)
     set color colorvec
   ]
   set l-index l-index + 1
  ]
  ask turtles [
    if dropped? [set color [127 127 127]]
  ]
  ask links [
    if ([dropped?] of end1) or ([dropped?] of end2) [set color [127 127 127 196]]
  ]
end

;; Set the local learning rate coefficients for the nodes
to set-learncoeffs
  let l-index ((length hiddensizes) + 1)
  let v (1 / (item (l-index - 2) hiddensizes))
  let lc (1 / ((item (l-index - 2) hiddensizes) * sqrt v))
  
  ask output-nodes [
    set estvar v
    set learncoeff lc
  ]
  
  set l-index (l-index - 1)
  
  repeat (length hiddensizes) - 1 [
    set v (((count hidden-nodes with [layer = (l-index - 1)]) * v) / (item (l-index - 2) hiddensizes))
    set lc (1 / ((item (l-index - 2) hiddensizes) * sqrt v))
    
    ask hidden-nodes with [layer = l-index] [
      set estvar v
      set learncoeff lc
    ]
    
    set l-index (l-index - 1)
  ]
  
  set v (((count input-nodes) * v) / (count input-nodes))
  set lc (1 / ((count input-nodes) * (sqrt v)))
  
  ask hidden-nodes with [layer = l-index] [
    set estvar v
    set learncoeff lc
  ]
end
    

;;;
;;; VISUAL LAYOUT FUNCTIONS
;;;

;; Find the appropriate x coordinate for this layer
to-report nodex [l-index]
  report min-pxcor + (((l-index + 1) * (world-width - 1)) / (length hiddensizes + 3))
end

;; Find the appropriate y cooridinate for this node
to-report nodey [l-index index in-layer]
  report max-pycor - (((index + 1) * (world-height - 1)) / (in-layer + 1))
end

;;;
;;; TRAINING PROCEDURES
;;;

to train
  set epoch-error 0
  let currentdatum (one-of traindata)
  let sortin sort input-nodes
  let index 0
  let target n-values 10 [ifelse-value (? = (item 0 currentdatum)) [1][-1]]
  
  repeat examples-per-epoch [
    if dropout? [
      ask hidden-nodes [if (random 1000) < dropout-rate [set dropped? true]]
    ]
    repeat (length sortin) [
      ask (item index sortin) [set activation ((item (index + 1) currentdatum) / 127.5) - 1]
      set index index + 1
    ]
    propagate
    back-propagate  target
    set index 0
    set currentdatum (one-of traindata)
    set target n-values 10 [ifelse-value (? = (item 0 currentdatum)) [1][-1]]
    ask hidden-nodes [set dropped? false]
  ]
  set epoch-error epoch-error / examples-per-epoch
  tick
end

;;;
;;; PROPAGATION PROCEDURES
;;;

;; carry out one calculation from beginning to end
to propagate
  let l-index 1
  repeat length hiddensizes [
    ask hidden-nodes with [layer = l-index and not dropped?] [ set activation new-activation ]
    set l-index l-index + 1
  ]
  ask output-nodes [set activation new-activation]
  recolor
end

;; Determine the activation of a node based on the activation of its input nodes
to-report new-activation  ;; node procedure
  report tanh sum [[activation] of end1 * weight] of my-in-links with [not [dropped?] of end1]
end

;; changes weights to correct for errors
to back-propagate [target]
  let example-error 0
  let sortout sort output-nodes
  let l-index (length hiddensizes) + 1
  let index 0
  
  repeat (count output-nodes) [
    ask (item index sortout) [
      set err (item index target) - activation
      set example-error example-error + (err ^ 2)
    ]
    set index index + 1
  ]
  
  set example-error .5 * example-error
  set l-index l-index - 1
  
  repeat length hiddensizes [
    ask hidden-nodes with [layer = l-index and not dropped?] [
      let sumerror sum [weight * ([err] of end2)] of my-out-links
      set err (1 - (activation ^ 2)) * sumerror
    ]
    set l-index l-index + 1
  ]
  
  ask links with [not [dropped?] of end1 and not [dropped?] of end2] [
    let change ([err * learncoeff * learning-rate] of end2)*([activation] of end1)
    set weight weight + change
  ]
  
  set epoch-error epoch-error + example-error
end

;;;
;;; MISC PROCEDURES
;;;

;; Calculates the tanh function
to-report tanh [input]
  let exp2x e ^ (2 * input)
  report (exp2x - 1) / (exp2x + 1)
end

;; computes the step function given an input value and the weight on the link
to-report step [input]
  report ifelse-value (input > 0) [1][0]
end


; Copyright 2006 Uri Wilensky.
; See Info tab for full copyright and license.
@#$#@#$#@
GRAPHICS-WINDOW
230
10
978
419
20
10
18.0
1
10
1
1
1
0
0
0
1
-20
20
-10
10
0
0
1
ticks
30.0

BUTTON
115
35
200
68
init-net
init-net
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

BUTTON
20
90
105
125
train
train
T
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SLIDER
14
128
220
161
learning-rate
learning-rate
0.0
.01
0.01
1.0E-4
1
NIL
HORIZONTAL

PLOT
10
275
220
430
Error vs. Epochs
Epochs
Error
0.0
10.0
0.0
1.0
true
false
"" ""
PENS
"default" 1.0 0 -16777216 true "" "plot epoch-error"

SLIDER
14
168
220
201
examples-per-epoch
examples-per-epoch
1.0
1000.0
22
1.0
1
NIL
HORIZONTAL

TEXTBOX
10
20
127
38
1. Setup neural net:
11
0.0
1

TEXTBOX
10
70
119
98
2. Train neural net:
11
0.0
1

BUTTON
120
90
205
123
train once
train
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

INPUTBOX
15
205
220
265
hidden-sizes-string
[25 5 5 10] 
1
0
String

BUTTON
20
35
97
68
NIL
setup
NIL
1
T
OBSERVER
NIL
NIL
NIL
NIL
1

SWITCH
235
430
337
463
dropout?
dropout?
0
1
-1000

SLIDER
350
430
522
463
dropout-rate
dropout-rate
0
1000
503
1
1
NIL
HORIZONTAL

@#$#@#$#@
## WHAT IS IT?

This is a model of arbitrarily large neural networks.  It is based on the Multilayer Perceptron model, but the network architecture is user-determined. 

This network is intended to provide a visualization of the process of neural network
training, and to serve as a platform for experimentation with an eye on qualitiative
intuitions.

## HOW IT WORKS

Initially the weights on the links of the networks are random, with some scaling based on the surrounding network.  

The nodes in the leftmost layer are the called the input nodes, the nodes in the middle layers are called the hidden nodes, and the nodes on the rightmost layer are called the output nodes.

The activation values of the input nodes are the inputs to the network. The activation values of the hidden nodes and output nodes are equal to the activation values of the layer before them, multiplied by their link weights, summed together, and passed through the tanh function.  The output of the network is 1 if the activation of the output node is greater than 0 and -1 if it is less than 0.

The tanh function maps negative values to values between -1 and 0, and maps positive values to values between 0 and 1.  The values increase nonlinearly between -1 and 1 with a sharp transition at 0. 

To train the network a lot of inputs are presented to the network along with how the network should correctly classify the inputs.  The network uses a back-propagation algorithm to pass error back from the output node and uses this error to update the weights along each link.

If dropout learning is enabled, hidden nodes are randomly dropped from training at each training step. This prevents the model from "overfitting", which means it stops it from making too large of assumptions about the entire data set based on the data it has seen so far.

## HOW TO USE IT

To use it press SETUP to load the training data and initialize the patches.

Enter a string of the form "["-any number of space seperated positive integers-"]" into HIDDEN-SIZES-STRING to set the number of hidden nodes in each hidden layer (and how many such layers there are).

Press INIT-NET to initialize the network.

Press TRAIN ONCE to run one epoch of training.  The number of examples presented to the network during this epoch is controlled by EXAMPLES-PER-EPOCH slider.

Press TRAIN to continually train the network.

In the view, the greater the intensity of the link's color the greater (in terms of absolute value) the weight it has.  If the link is red then it has a positive weight.  If the link is blue then it has a negative weight.

LEARNING-RATE controls how much the neural network will learn from any one example.

DROPOUT? activates dropout learning

DROPOUT-RATE controls the probability that the hidden nodes drop out at each step, out of 1000.

## THINGS TO NOTICE

As the network trains, high-weight edges (intuitively, connections that the model is placing importance on) become brighter. This exposes the actual process of learning in an intuitive visual medium, and can sometimes be extremely informative.

## THINGS TO TRY

Manipulate the HIDDEN-SIZES-STRING. What happens to the learning rate as the number of nodes increases? What happens to the accuracy? How does this effect the visualization of the model?

## EXTENDING THE MODEL

The ability to set a threshold weight under which edges become transparent could make the model visualization less cluttered and easier to interpret.

The model would be a viable candidate for extension by any standard neural network optimization, which could be compared to normal learning in a similar way to how dropout was handled in this case.

## NETLOGO FEATURES

This model uses the link primitives.  It also makes heavy use of lists. Additionally, the csv extension is used to load the MNIST data set.



## COPYRIGHT AND LICENSE

Copyright 2006 Uri Wilensky.

![CC BY-NC-SA 3.0](http://ccl.northwestern.edu/images/creativecommons/byncsa.png)

This work is licensed under the Creative Commons Attribution-NonCommercial-ShareAlike 3.0 License.  To view a copy of this license, visit http://creativecommons.org/licenses/by-nc-sa/3.0/ or send a letter to Creative Commons, 559 Nathan Abbott Way, Stanford, California 94305, USA.

Commercial licenses are also available. To inquire about commercial licenses, please contact Uri Wilensky at uri@northwestern.edu.
@#$#@#$#@
default
true
0
Polygon -7500403 true true 150 5 40 250 150 205 260 250

airplane
true
0
Polygon -7500403 true true 150 0 135 15 120 60 120 105 15 165 15 195 120 180 135 240 105 270 120 285 150 270 180 285 210 270 165 240 180 180 285 195 285 165 180 105 180 60 165 15

arrow
true
0
Polygon -7500403 true true 150 0 0 150 105 150 105 293 195 293 195 150 300 150

bias-node
false
0
Circle -16777216 true false 0 0 300
Circle -7500403 true true 30 30 240
Polygon -16777216 true false 120 60 150 60 165 60 165 225 180 225 180 240 135 240 135 225 150 225 150 75 135 75 150 60

box
false
0
Polygon -7500403 true true 150 285 285 225 285 75 150 135
Polygon -7500403 true true 150 135 15 75 150 15 285 75
Polygon -7500403 true true 15 75 15 225 150 285 150 135
Line -16777216 false 150 285 150 135
Line -16777216 false 150 135 15 75
Line -16777216 false 150 135 285 75

bug
true
0
Circle -7500403 true true 96 182 108
Circle -7500403 true true 110 127 80
Circle -7500403 true true 110 75 80
Line -7500403 true 150 100 80 30
Line -7500403 true 150 100 220 30

butterfly
true
0
Polygon -7500403 true true 150 165 209 199 225 225 225 255 195 270 165 255 150 240
Polygon -7500403 true true 150 165 89 198 75 225 75 255 105 270 135 255 150 240
Polygon -7500403 true true 139 148 100 105 55 90 25 90 10 105 10 135 25 180 40 195 85 194 139 163
Polygon -7500403 true true 162 150 200 105 245 90 275 90 290 105 290 135 275 180 260 195 215 195 162 165
Polygon -16777216 true false 150 255 135 225 120 150 135 120 150 105 165 120 180 150 165 225
Circle -16777216 true false 135 90 30
Line -16777216 false 150 105 195 60
Line -16777216 false 150 105 105 60

car
false
0
Polygon -7500403 true true 300 180 279 164 261 144 240 135 226 132 213 106 203 84 185 63 159 50 135 50 75 60 0 150 0 165 0 225 300 225 300 180
Circle -16777216 true false 180 180 90
Circle -16777216 true false 30 180 90
Polygon -16777216 true false 162 80 132 78 134 135 209 135 194 105 189 96 180 89
Circle -7500403 true true 47 195 58
Circle -7500403 true true 195 195 58

circle
false
0
Circle -7500403 true true 0 0 300

circle 2
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240

cow
false
0
Polygon -7500403 true true 200 193 197 249 179 249 177 196 166 187 140 189 93 191 78 179 72 211 49 209 48 181 37 149 25 120 25 89 45 72 103 84 179 75 198 76 252 64 272 81 293 103 285 121 255 121 242 118 224 167
Polygon -7500403 true true 73 210 86 251 62 249 48 208
Polygon -7500403 true true 25 114 16 195 9 204 23 213 25 200 39 123

cylinder
false
0
Circle -7500403 true true 0 0 300

dot
false
0
Circle -7500403 true true 90 90 120

face happy
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 255 90 239 62 213 47 191 67 179 90 203 109 218 150 225 192 218 210 203 227 181 251 194 236 217 212 240

face neutral
false
0
Circle -7500403 true true 8 7 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Rectangle -16777216 true false 60 195 240 225

face sad
false
0
Circle -7500403 true true 8 8 285
Circle -16777216 true false 60 75 60
Circle -16777216 true false 180 75 60
Polygon -16777216 true false 150 168 90 184 62 210 47 232 67 244 90 220 109 205 150 198 192 205 210 220 227 242 251 229 236 206 212 183

fish
false
0
Polygon -1 true false 44 131 21 87 15 86 0 120 15 150 0 180 13 214 20 212 45 166
Polygon -1 true false 135 195 119 235 95 218 76 210 46 204 60 165
Polygon -1 true false 75 45 83 77 71 103 86 114 166 78 135 60
Polygon -7500403 true true 30 136 151 77 226 81 280 119 292 146 292 160 287 170 270 195 195 210 151 212 30 166
Circle -16777216 true false 215 106 30

flag
false
0
Rectangle -7500403 true true 60 15 75 300
Polygon -7500403 true true 90 150 270 90 90 30
Line -7500403 true 75 135 90 135
Line -7500403 true 75 45 90 45

flower
false
0
Polygon -10899396 true false 135 120 165 165 180 210 180 240 150 300 165 300 195 240 195 195 165 135
Circle -7500403 true true 85 132 38
Circle -7500403 true true 130 147 38
Circle -7500403 true true 192 85 38
Circle -7500403 true true 85 40 38
Circle -7500403 true true 177 40 38
Circle -7500403 true true 177 132 38
Circle -7500403 true true 70 85 38
Circle -7500403 true true 130 25 38
Circle -7500403 true true 96 51 108
Circle -16777216 true false 113 68 74
Polygon -10899396 true false 189 233 219 188 249 173 279 188 234 218
Polygon -10899396 true false 180 255 150 210 105 210 75 240 135 240

house
false
0
Rectangle -7500403 true true 45 120 255 285
Rectangle -16777216 true false 120 210 180 285
Polygon -7500403 true true 15 120 150 15 285 120
Line -16777216 false 30 120 270 120

leaf
false
0
Polygon -7500403 true true 150 210 135 195 120 210 60 210 30 195 60 180 60 165 15 135 30 120 15 105 40 104 45 90 60 90 90 105 105 120 120 120 105 60 120 60 135 30 150 15 165 30 180 60 195 60 180 120 195 120 210 105 240 90 255 90 263 104 285 105 270 120 285 135 240 165 240 180 270 195 240 210 180 210 165 195
Polygon -7500403 true true 135 195 135 240 120 255 105 255 105 285 135 285 165 240 165 195

line
true
0
Line -7500403 true 150 0 150 300

line half
true
0
Line -7500403 true 150 0 150 150

link
true
0
Line -7500403 true 150 0 150 300

link direction
true
0

output-node
false
1
Circle -7500403 true false 0 0 300
Circle -2674135 true true 30 30 240
Polygon -7500403 true false 195 75 90 75 150 150 90 225 195 225 195 210 195 195 180 210 120 210 165 150 120 90 180 90 195 105 195 75

pentagon
false
0
Polygon -7500403 true true 150 15 15 120 60 285 240 285 285 120

person
false
0
Circle -7500403 true true 110 5 80
Polygon -7500403 true true 105 90 120 195 90 285 105 300 135 300 150 225 165 300 195 300 210 285 180 195 195 90
Rectangle -7500403 true true 127 79 172 94
Polygon -7500403 true true 195 90 240 150 225 180 165 105
Polygon -7500403 true true 105 90 60 150 75 180 135 105

plant
false
0
Rectangle -7500403 true true 135 90 165 300
Polygon -7500403 true true 135 255 90 210 45 195 75 255 135 285
Polygon -7500403 true true 165 255 210 210 255 195 225 255 165 285
Polygon -7500403 true true 135 180 90 135 45 120 75 180 135 210
Polygon -7500403 true true 165 180 165 210 225 180 255 120 210 135
Polygon -7500403 true true 135 105 90 60 45 45 75 105 135 135
Polygon -7500403 true true 165 105 165 135 225 105 255 45 210 60
Polygon -7500403 true true 135 90 120 45 150 15 180 45 165 90

square
false
0
Rectangle -7500403 true true 30 30 270 270

square 2
false
0
Rectangle -7500403 true true 30 30 270 270
Rectangle -16777216 true false 60 60 240 240

star
false
0
Polygon -7500403 true true 151 1 185 108 298 108 207 175 242 282 151 216 59 282 94 175 3 108 116 108

target
false
0
Circle -7500403 true true 0 0 300
Circle -16777216 true false 30 30 240
Circle -7500403 true true 60 60 180
Circle -16777216 true false 90 90 120
Circle -7500403 true true 120 120 60

tree
false
0
Circle -7500403 true true 118 3 94
Rectangle -6459832 true false 120 195 180 300
Circle -7500403 true true 65 21 108
Circle -7500403 true true 116 41 127
Circle -7500403 true true 45 90 120
Circle -7500403 true true 104 74 152

triangle
false
0
Polygon -7500403 true true 150 30 15 255 285 255

triangle 2
false
0
Polygon -7500403 true true 150 30 15 255 285 255
Polygon -16777216 true false 151 99 225 223 75 224

truck
false
0
Rectangle -7500403 true true 4 45 195 187
Polygon -7500403 true true 296 193 296 150 259 134 244 104 208 104 207 194
Rectangle -1 true false 195 60 195 105
Polygon -16777216 true false 238 112 252 141 219 141 218 112
Circle -16777216 true false 234 174 42
Rectangle -7500403 true true 181 185 214 194
Circle -16777216 true false 144 174 42
Circle -16777216 true false 24 174 42
Circle -7500403 false true 24 174 42
Circle -7500403 false true 144 174 42
Circle -7500403 false true 234 174 42

turtle
true
0
Polygon -10899396 true false 215 204 240 233 246 254 228 266 215 252 193 210
Polygon -10899396 true false 195 90 225 75 245 75 260 89 269 108 261 124 240 105 225 105 210 105
Polygon -10899396 true false 105 90 75 75 55 75 40 89 31 108 39 124 60 105 75 105 90 105
Polygon -10899396 true false 132 85 134 64 107 51 108 17 150 2 192 18 192 52 169 65 172 87
Polygon -10899396 true false 85 204 60 233 54 254 72 266 85 252 107 210
Polygon -7500403 true true 119 75 179 75 209 101 224 135 220 225 175 261 128 261 81 224 74 135 88 99

wheel
false
0
Circle -7500403 true true 3 3 294
Circle -16777216 true false 30 30 240
Line -7500403 true 150 285 150 15
Line -7500403 true 15 150 285 150
Circle -7500403 true true 120 120 60
Line -7500403 true 216 40 79 269
Line -7500403 true 40 84 269 221
Line -7500403 true 40 216 269 79
Line -7500403 true 84 40 221 269

x
false
0
Polygon -7500403 true true 270 75 225 30 30 225 75 270
Polygon -7500403 true true 30 75 75 30 270 225 225 270

@#$#@#$#@
NetLogo 5.2.0-RC5
@#$#@#$#@
setup repeat 100 [ train ]
@#$#@#$#@
@#$#@#$#@
<experiments>
  <experiment name="DropoutComparison" repetitions="3" runMetricsEveryStep="true">
    <setup>init-net</setup>
    <go>train</go>
    <timeLimit steps="50"/>
    <metric>epoch-error</metric>
    <enumeratedValueSet variable="dropout">
      <value value="true"/>
      <value value="false"/>
    </enumeratedValueSet>
    <enumeratedValueSet variable="examples-per-epoch">
      <value value="25"/>
    </enumeratedValueSet>
    <steppedValueSet variable="dropout-rate" first="0" step="50" last="750"/>
    <enumeratedValueSet variable="hidden-sizes-string">
      <value value="&quot;[100 50 50 50] &quot;"/>
    </enumeratedValueSet>
    <steppedValueSet variable="learning-rate" first="0.0025" step="0.0025" last="0.01"/>
  </experiment>
</experiments>
@#$#@#$#@
@#$#@#$#@
default
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0

small-arrow-shape
0.0
-0.2 0 0.0 1.0
0.0 1 1.0 0.0
0.2 0 0.0 1.0
link direction
true
0
Line -7500403 true 150 150 135 180
Line -7500403 true 150 150 165 180

@#$#@#$#@
1
@#$#@#$#@
